<?php
require_once '../config/config.php';
require_once '../includes/security.php';
require_once '../includes/authorization.php';
require_once '../includes/error_handler.php';
require_once '../includes/user_manager.php';
require_once '../config/database.php';

requireLogin();

Authorization::requireAdmin();

$userManager = new UserManager();
$message = '';
$messageType = 'success';

if ($_POST) {
    try {
        if (!Security::validateCSRFToken($_POST['csrf_token'] ?? '')) {
            throw new Exception('Güvenlik hatası: Geçersiz token');
        }
        
        $action = $_POST['action'] ?? '';
        
        switch ($action) {
            case 'create_user':
                $kullanici_adi = Security::sanitizeInput($_POST['kullanici_adi'] ?? '');
                $sifre = $_POST['sifre'] ?? '';
                $ad_soyad = Security::sanitizeInput($_POST['ad_soyad'] ?? '');
                $rol_id = (int)($_POST['rol_id'] ?? 0);
                $aktif = isset($_POST['aktif']) ? 1 : 0;
                
                if (empty($kullanici_adi) || empty($sifre) || empty($ad_soyad) || $rol_id <= 0) {
                    throw new Exception('Tüm alanları doldurunuz');
                }
                
                if (strlen($sifre) < 6) {
                    throw new Exception('Şifre en az 6 karakter olmalıdır');
                }
                
                $result = $userManager->createUser($kullanici_adi, $sifre, $ad_soyad, $rol_id, $aktif);
                $message = $result['message'];
                $messageType = $result['success'] ? 'success' : 'danger';
                break;
                
            case 'update_user':
                $kullanici_id = (int)($_POST['kullanici_id'] ?? 0);
                $kullanici_adi = Security::sanitizeInput($_POST['kullanici_adi'] ?? '');
                $ad_soyad = Security::sanitizeInput($_POST['ad_soyad'] ?? '');
                $rol_id = (int)($_POST['rol_id'] ?? 0);
                $aktif = isset($_POST['aktif']) ? 1 : 0;
                $yeni_sifre = $_POST['yeni_sifre'] ?? '';
                
                if ($kullanici_id <= 0 || empty($kullanici_adi) || empty($ad_soyad) || $rol_id <= 0) {
                    throw new Exception('Tüm alanları doldurunuz');
                }
                
                if (!empty($yeni_sifre) && strlen($yeni_sifre) < 6) {
                    throw new Exception('Şifre en az 6 karakter olmalıdır');
                }
                
                $result = $userManager->updateUser($kullanici_id, $kullanici_adi, $ad_soyad, $rol_id, $aktif, $yeni_sifre ?: null);
                $message = $result['message'];
                $messageType = $result['success'] ? 'success' : 'danger';
                break;
                
            case 'delete_user':
                $kullanici_id = (int)($_POST['kullanici_id'] ?? 0);
                
                if ($kullanici_id <= 0) {
                    throw new Exception('Geçersiz kullanıcı ID');
                }
                
                $result = $userManager->deleteUser($kullanici_id);
                $message = $result['message'];
                $messageType = $result['success'] ? 'success' : 'danger';
                break;
                
            case 'change_password':
                $kullanici_id = $_SESSION['user_id'];
                $eski_sifre = $_POST['eski_sifre'] ?? '';
                $yeni_sifre = $_POST['yeni_sifre'] ?? '';
                $yeni_sifre_tekrar = $_POST['yeni_sifre_tekrar'] ?? '';
                
                if (empty($eski_sifre) || empty($yeni_sifre) || empty($yeni_sifre_tekrar)) {
                    throw new Exception('Tüm şifre alanlarını doldurunuz');
                }
                
                if ($yeni_sifre !== $yeni_sifre_tekrar) {
                    throw new Exception('Yeni şifreler eşleşmiyor');
                }
                
                if (strlen($yeni_sifre) < 6) {
                    throw new Exception('Şifre en az 6 karakter olmalıdır');
                }
                
                $result = $userManager->changePassword($kullanici_id, $eski_sifre, $yeni_sifre);
                $message = $result['message'];
                $messageType = $result['success'] ? 'success' : 'danger';
                break;
        }
        
    } catch (Exception $e) {
        $message = $e->getMessage();
        $messageType = 'danger';
        
        ErrorHandler::logError('user_management', $e->getMessage(), [
            'action' => $action ?? 'unknown',
            'user_id' => $_SESSION['user_id'] ?? 0,
            'ip' => Security::getClientIP()
        ]);
    }
}

$users = $userManager->getAllUsers(true); // Sadece aktif kullanıcıları göster
$roles = $userManager->getAllRoles();
$stats = $userManager->getUserStats();

$pageTitle = 'Kullanıcı Yönetimi';
require_once '../includes/header.php';
?>

<?php require_once '../includes/sidebar.php'; ?>

<div class="main-content">
    <div class="container-fluid">
        <main class="px-md-4">
            <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                <h1 class="h2"><i class="fas fa-users me-2"></i>Kullanıcı Yönetimi</h1>
                <div class="btn-toolbar mb-2 mb-md-0">
                    <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#createUserModal">
                        <i class="fas fa-plus me-2"></i>Yeni Kullanıcı
                    </button>
                </div>
            </div>

            <?php if ($message): ?>
            <div class="alert alert-<?= $messageType ?> alert-dismissible fade show" role="alert">
                <?= htmlspecialchars($message) ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
            <?php endif; ?>

            
            <div class="row mb-4">
                <div class="col-md-3">
                    <div class="card text-white bg-primary">
                        <div class="card-body">
                            <div class="d-flex justify-content-between">
                                <div>
                                    <h4><?= $stats['toplam_kullanici'] ?? 0 ?></h4>
                                    <p class="card-text">Toplam Kullanıcı</p>
                                </div>
                                <div class="align-self-center">
                                    <i class="fas fa-users fa-2x"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card text-white bg-success">
                        <div class="card-body">
                            <div class="d-flex justify-content-between">
                                <div>
                                    <h4><?= $stats['aktif_kullanici'] ?? 0 ?></h4>
                                    <p class="card-text">Aktif Kullanıcı</p>
                                </div>
                                <div class="align-self-center">
                                    <i class="fas fa-user-check fa-2x"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-md-6">
                    <div class="card">
                        <div class="card-header">
                            <h6><i class="fas fa-chart-pie me-2"></i>Rol Dağılımı</h6>
                        </div>
                        <div class="card-body">
                            <div class="row">
                                <?php foreach ($stats['rol_dagilimi'] ?? [] as $rol): ?>
                                <div class="col-6 mb-2">
                                    <span class="badge bg-secondary me-2"><?= $rol['sayi'] ?></span>
                                    <?= htmlspecialchars($rol['rol_adi']) ?>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            
            <div class="card">
                <div class="card-header">
                    <h5><i class="fas fa-list me-2"></i>Kullanıcı Listesi</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead>
                                <tr>
                                    <th>Kullanıcı Adı</th>
                                    <th>Ad Soyad</th>
                                    <th>Rol</th>
                                    <th>Durum</th>
                                    <th>Oluşturulma Tarihi</th>
                                    <th>İşlemler</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($users as $user): ?>
                                <tr>
                                    <td><?= htmlspecialchars($user['kullanici_adi']) ?></td>
                                    <td><?= htmlspecialchars($user['ad_soyad']) ?></td>
                                    <td>
                                        <span class="badge bg-<?= $user['rol_adi'] == 'admin' ? 'danger' : ($user['rol_adi'] == 'manager' ? 'warning' : 'info') ?>">
                                            <?= htmlspecialchars($user['rol_adi']) ?>
                                        </span>
                                    </td>
                                    <td>
                                        <span class="badge bg-<?= $user['aktif'] ? 'success' : 'secondary' ?>">
                                            <?= $user['aktif'] ? 'Aktif' : 'Pasif' ?>
                                        </span>
                                    </td>
                                    <td><?= date('d.m.Y H:i', strtotime($user['olusturulma_tarihi'])) ?></td>
                                    <td>
                                        <button type="button" class="btn btn-sm btn-outline-primary" 
                                                onclick="editUser(<?= htmlspecialchars(json_encode($user)) ?>)">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <?php if ($user['kullanici_id'] != $_SESSION['user_id']): ?>
                                        <button type="button" class="btn btn-sm btn-outline-danger" 
                                                onclick="deleteUser(<?= $user['kullanici_id'] ?>, '<?= htmlspecialchars($user['kullanici_adi']) ?>')">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </main>
    </div>
</div>


<div class="modal fade" id="createUserModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Yeni Kullanıcı Oluştur</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <input type="hidden" name="csrf_token" value="<?= Security::generateCSRFToken() ?>">
                    <input type="hidden" name="action" value="create_user">
                    
                    <div class="mb-3">
                        <label for="kullanici_adi" class="form-label">Kullanıcı Adı *</label>
                        <input type="text" class="form-control" id="kullanici_adi" name="kullanici_adi" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="sifre" class="form-label">Şifre *</label>
                        <input type="password" class="form-control" id="sifre" name="sifre" required minlength="6">
                        <div class="form-text">En az 6 karakter olmalıdır</div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="ad_soyad" class="form-label">Ad Soyad *</label>
                        <input type="text" class="form-control" id="ad_soyad" name="ad_soyad" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="rol_id" class="form-label">Rol *</label>
                        <select class="form-select" id="rol_id" name="rol_id" required>
                            <option value="">Rol Seçin</option>
                            <?php foreach ($roles as $role): ?>
                            <option value="<?= $role['rol_id'] ?>"><?= htmlspecialchars($role['rol_adi']) ?> - <?= htmlspecialchars($role['aciklama']) ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="mb-3 form-check">
                        <input type="checkbox" class="form-check-input" id="aktif" name="aktif" checked>
                        <label class="form-check-label" for="aktif">Aktif</label>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">İptal</button>
                    <button type="submit" class="btn btn-primary">Oluştur</button>
                </div>
            </form>
        </div>
    </div>
</div>


<div class="modal fade" id="editUserModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Kullanıcı Düzenle</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <input type="hidden" name="csrf_token" value="<?= Security::generateCSRFToken() ?>">
                    <input type="hidden" name="action" value="update_user">
                    <input type="hidden" name="kullanici_id" id="edit_kullanici_id">
                    
                    <div class="mb-3">
                        <label for="edit_kullanici_adi" class="form-label">Kullanıcı Adı *</label>
                        <input type="text" class="form-control" id="edit_kullanici_adi" name="kullanici_adi" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="edit_ad_soyad" class="form-label">Ad Soyad *</label>
                        <input type="text" class="form-control" id="edit_ad_soyad" name="ad_soyad" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="edit_rol_id" class="form-label">Rol *</label>
                        <select class="form-select" id="edit_rol_id" name="rol_id" required>
                            <?php foreach ($roles as $role): ?>
                            <option value="<?= $role['rol_id'] ?>"><?= htmlspecialchars($role['rol_adi']) ?> - <?= htmlspecialchars($role['aciklama']) ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="yeni_sifre" class="form-label">Yeni Şifre</label>
                        <input type="password" class="form-control" id="yeni_sifre" name="yeni_sifre" minlength="6">
                        <div class="form-text">Boş bırakırsanız şifre değişmez</div>
                    </div>
                    
                    <div class="mb-3 form-check">
                        <input type="checkbox" class="form-check-input" id="edit_aktif" name="aktif">
                        <label class="form-check-label" for="edit_aktif">Aktif</label>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">İptal</button>
                    <button type="submit" class="btn btn-primary">Güncelle</button>
                </div>
            </form>
        </div>
    </div>
</div>


<div class="modal fade" id="deleteUserModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Kullanıcı Sil</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <input type="hidden" name="csrf_token" value="<?= Security::generateCSRFToken() ?>">
                    <input type="hidden" name="action" value="delete_user">
                    <input type="hidden" name="kullanici_id" id="delete_kullanici_id">
                    
                    <p>Bu kullanıcıyı silmek istediğinizden emin misiniz?</p>
                    <p><strong id="delete_kullanici_adi"></strong></p>
                    <p class="text-muted">Bu işlem geri alınamaz.</p>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">İptal</button>
                    <button type="submit" class="btn btn-danger">Sil</button>
                </div>
            </form>
        </div>
    </div>
</div>

        </main>
    </div>
</div>

<script>
function editUser(user) {
    document.getElementById('edit_kullanici_id').value = user.kullanici_id;
    document.getElementById('edit_kullanici_adi').value = user.kullanici_adi;
    document.getElementById('edit_ad_soyad').value = user.ad_soyad;
    document.getElementById('edit_rol_id').value = user.rol_id;
    document.getElementById('edit_aktif').checked = user.aktif == 1;
    
    new bootstrap.Modal(document.getElementById('editUserModal')).show();
}

function deleteUser(kullanici_id, kullanici_adi) {
    document.getElementById('delete_kullanici_id').value = kullanici_id;
    document.getElementById('delete_kullanici_adi').textContent = kullanici_adi;
    
    new bootstrap.Modal(document.getElementById('deleteUserModal')).show();
}
</script>

<?php require_once '../includes/footer.php'; ?>