<?php
require_once '../config/config.php';
require_once '../includes/security.php';
require_once '../includes/authorization.php';

requireLogin();

Authorization::requirePageAccess('reports', 'okuma');

$error_message = '';
$success_message = '';

$search = Security::sanitizeInput($_GET['search'] ?? '');
$cari_tipi = Security::sanitizeInput($_GET['cari_tipi'] ?? '');
$tarih_baslangic = Security::sanitizeInput($_GET['tarih_baslangic'] ?? '');
$tarih_bitis = Security::sanitizeInput($_GET['tarih_bitis'] ?? '');
$bakiye_durumu = Security::sanitizeInput($_GET['bakiye_durumu'] ?? '');
$page = max(1, (int)($_GET['page'] ?? 1));
$per_page = 20;
$offset = ($page - 1) * $per_page;

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['odeme_yap'])) {
    try {
        validateCSRF();
        
        $cari_id = (int)($_POST['cari_id'] ?? 0);
        $odeme_tutari = (float)($_POST['odeme_tutari'] ?? 0);
        $aciklama = Security::sanitizeInput($_POST['aciklama'] ?? '');
        
        if ($cari_id <= 0) {
            throw new Exception('Geçersiz cari.');
        }
        
        if ($odeme_tutari <= 0) {
            throw new Exception('Ödeme tutarı pozitif olmalıdır.');
        }
        
        // Transaction başlat
        $database->beginTransaction();
        
        $sql = "SELECT * FROM cariler WHERE cari_id = ?";
        $customer = $database->fetch($sql, [$cari_id]);
        if (!$customer) {
            throw new Exception('Cari bulunamadı.');
        }
        
        // Müşteri ve tedarikçi için farklı kontroller
        if ($customer['cari_tipi'] === 'musteri') {
            if ($odeme_tutari > $customer['bakiye']) {
                throw new Exception('Ödeme tutarı bakiyeden fazla olamaz.');
            }
        } else { // tedarikci
            if ($odeme_tutari > $customer['bakiye']) {
                throw new Exception('Ödeme tutarı borçtan fazla olamaz.');
            }
        }
        
        // Merkezi BalanceManager kullanarak ödeme işlemini kaydet
        require_once '../includes/balance_manager.php';
        $balanceManager = new BalanceManager($database);
        
        $odeme_tipi = ($customer['cari_tipi'] === 'musteri') ? 'tahsilat' : 'odeme';
        $result = $balanceManager->processManualPayment($cari_id, $odeme_tutari, $odeme_tipi, $aciklama);
        
        if (!$result['success']) {
            throw new Exception($result['message']);
        }
        
        // Ödenmemiş satışları güncelle (FIFO mantığı ile)
        $sql = "SELECT satis_id, kalan_tutar FROM satislar WHERE cari_id = ? AND kalan_tutar > 0 ORDER BY satis_tarihi ASC";
        $unpaid_sales = $database->fetchAll($sql, [$cari_id]);
        
        $remaining_payment = $odeme_tutari;
        foreach ($unpaid_sales as $sale) {
            if ($remaining_payment <= 0) break;
            
            $payment_for_this_sale = min($remaining_payment, $sale['kalan_tutar']);
            
            // Satışın kalan tutarını güncelle
            $sql = "UPDATE satislar SET 
                    kalan_tutar = kalan_tutar - ?, 
                    odenen_tutar = odenen_tutar + ? 
                    WHERE satis_id = ?";
            $database->execute($sql, [$payment_for_this_sale, $payment_for_this_sale, $sale['satis_id']]);
            
            $remaining_payment -= $payment_for_this_sale;
        }
        
        // Transaction commit
        $database->commit();
        
        // POST/Redirect/GET pattern - F5 ile çift işlem önleme
        $_SESSION['success_message'] = 'Ödeme başarıyla kaydedildi.';
        header('Location: reports.php?' . http_build_query($_GET));
        exit();
        
    } catch (Exception $e) {
        // Transaction rollback
        if ($database->inTransaction()) {
            $database->rollback();
        }
        $error_message = $e->getMessage();
        error_log("Ödeme işlemi hatası: " . $e->getMessage());
    }
}

// Session'dan success mesajını al ve temizle
if (isset($_SESSION['success_message'])) {
    $success_message = $_SESSION['success_message'];
    unset($_SESSION['success_message']);
}

$where_conditions = ["1=1"];
$params = [];

if (!empty($search)) {
    $where_conditions[] = "(c.cari_adi LIKE ? OR c.telefon LIKE ? OR c.email LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

if (!empty($cari_tipi)) {
    $where_conditions[] = "c.cari_tipi = ?";
    $params[] = $cari_tipi;
}

if (!empty($bakiye_durumu)) {
    if ($bakiye_durumu === 'borclu') {
        $where_conditions[] = "c.bakiye > 0";
    } elseif ($bakiye_durumu === 'alacakli') {
        $where_conditions[] = "c.bakiye < 0";
    } elseif ($bakiye_durumu === 'sifir') {
        $where_conditions[] = "c.bakiye = 0";
    }
}

$where_clause = 'WHERE ' . implode(' AND ', $where_conditions);

$count_sql = "SELECT COUNT(*) as total FROM cariler c $where_clause";
$total_records = $database->fetch($count_sql, $params)['total'];
$total_pages = ceil($total_records / $per_page);

$sql = "SELECT c.*, 
               CASE 
                   WHEN c.cari_tipi = 'musteri' THEN COUNT(s.satis_id)
                   ELSE (SELECT COUNT(*) FROM cari_hareketler ch WHERE ch.cari_id = c.cari_id AND ch.hareket_tipi = 'borc')
               END as toplam_satis,
               CASE 
                   WHEN c.cari_tipi = 'musteri' THEN COALESCE(SUM(s.toplam_tutar), 0)
                   ELSE (SELECT COALESCE(SUM(ch.tutar), 0) FROM cari_hareketler ch WHERE ch.cari_id = c.cari_id AND ch.hareket_tipi = 'borc')
               END as toplam_alis_tutari,
               (SELECT COUNT(*) FROM cari_hareketler ch WHERE ch.cari_id = c.cari_id) as hareket_sayisi
        FROM cariler c
        LEFT JOIN satislar s ON c.cari_id = s.cari_id
        $where_clause
        GROUP BY c.cari_id
        ORDER BY c.bakiye DESC, c.cari_adi ASC
        LIMIT $per_page OFFSET $offset";

$customers = $database->fetchAll($sql, $params);

$stats_sql = "SELECT 
                COUNT(*) as toplam_cari,
                COUNT(CASE WHEN bakiye > 0 THEN 1 END) as borclu_cari,
                COUNT(CASE WHEN bakiye < 0 THEN 1 END) as alacakli_cari,
                COUNT(CASE WHEN bakiye = 0 THEN 1 END) as sifir_bakiye,
                COALESCE(SUM(CASE WHEN bakiye > 0 THEN bakiye END), 0) as toplam_borc,
                COALESCE(SUM(CASE WHEN bakiye < 0 THEN ABS(bakiye) END), 0) as toplam_alacak,
                
                -- Müşteri istatistikleri
                COUNT(CASE WHEN cari_tipi = 'musteri' THEN 1 END) as toplam_musteri,
                COUNT(CASE WHEN cari_tipi = 'musteri' AND bakiye > 0 THEN 1 END) as borclu_musteri,
                COALESCE(SUM(CASE WHEN cari_tipi = 'musteri' AND bakiye > 0 THEN bakiye END), 0) as musteri_borc,
                
                -- Tedarikçi istatistikleri
                COUNT(CASE WHEN cari_tipi = 'tedarikci' THEN 1 END) as toplam_tedarikci,
                COUNT(CASE WHEN cari_tipi = 'tedarikci' AND bakiye > 0 THEN 1 END) as borclu_tedarikci,
                COALESCE(SUM(CASE WHEN cari_tipi = 'tedarikci' AND bakiye > 0 THEN bakiye END), 0) as tedarikci_borc
              FROM cariler";
$stats = $database->fetch($stats_sql);

$sales_where = ["1=1"];
$sales_params = [];

if (!empty($tarih_baslangic)) {
    $sales_where[] = "s.satis_tarihi >= ?";
    $sales_params[] = $tarih_baslangic . ' 00:00:00';
}

if (!empty($tarih_bitis)) {
    $sales_where[] = "s.satis_tarihi <= ?";
    $sales_params[] = $tarih_bitis . ' 23:59:59';
}

$sales_where_clause = 'WHERE ' . implode(' AND ', $sales_where);

$recent_sales_sql = "SELECT s.*, c.cari_adi 
                     FROM satislar s
                     JOIN cariler c ON s.cari_id = c.cari_id
                     $sales_where_clause
                     ORDER BY s.satis_tarihi DESC
                     LIMIT 10";
$recent_sales = $database->fetchAll($recent_sales_sql, $sales_params);

$page_title = "Cari Takip ve Raporlar";
?>
<?php include '../includes/header.php'; ?>
    <style>
        .card {
            border: none;
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.08);
        }
        .btn {
            border-radius: 10px;
        }
        .form-control, .form-select {
            border-radius: 10px;
            border: 2px solid #e9ecef;
        }
        .form-control:focus, .form-select:focus {
            border-color: #667eea;
            box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.25);
        }
        .stats-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 15px;
        }
        .balance-positive {
            color: #dc3545;
            font-weight: bold;
        }
        .balance-negative {
            color: #198754;
            font-weight: bold;
        }
        .balance-zero {
            color: #6c757d;
        }

        .table-responsive {
            border-radius: 15px;
            overflow: hidden;
        }
        .pagination {
            justify-content: center;
        }
    </style>

<?php include '../includes/sidebar.php'; ?>

<main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
    
    <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
        <h1 class="h2">
            <button class="btn btn-outline-secondary d-md-none me-2" type="button" id="sidebarToggle">
                <i class="fas fa-bars"></i>
            </button>
            <?php echo $page_title; ?>
        </h1>
        <div class="btn-toolbar mb-2 mb-md-0">
            <a href="dashboard.php" class="btn btn-sm btn-outline-secondary">
                <i class="fas fa-arrow-left me-1"></i>Dashboard'a Dön
            </a>
        </div>
    </div>

    
    <div class="container-fluid">
            <?php if ($error_message): ?>
                <div class="alert alert-danger" role="alert">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    <?php echo Security::escapeOutput($error_message); ?>
                </div>
            <?php endif; ?>

            <?php if ($success_message): ?>
                <div class="alert alert-success" role="alert">
                    <i class="fas fa-check-circle me-2"></i>
                    <?php echo Security::escapeOutput($success_message); ?>
                </div>
            <?php endif; ?>

            
            <div class="row mb-4">
                <div class="col-md-3 mb-3">
                    <div class="card stats-card">
                        <div class="card-body text-center">
                            <i class="fas fa-users fa-2x mb-2"></i>
                            <h4><?php echo number_format($stats['toplam_cari']); ?></h4>
                            <p class="mb-0">Toplam Cari</p>
                            <small>Müşteri: <?php echo number_format($stats['toplam_musteri']); ?> | Tedarikçi: <?php echo number_format($stats['toplam_tedarikci']); ?></small>
                        </div>
                    </div>
                </div>
                <div class="col-md-3 mb-3">
                    <div class="card bg-danger text-white">
                        <div class="card-body text-center">
                            <i class="fas fa-user-friends fa-2x mb-2"></i>
                            <h4><?php echo number_format($stats['borclu_musteri']); ?></h4>
                            <p class="mb-0">Borçlu Müşteri</p>
                            <small>₺<?php echo number_format($stats['musteri_borc'], 2); ?></small>
                        </div>
                    </div>
                </div>
                <div class="col-md-3 mb-3">
                    <div class="card bg-warning text-white">
                        <div class="card-body text-center">
                            <i class="fas fa-truck fa-2x mb-2"></i>
                            <h4><?php echo number_format($stats['borclu_tedarikci']); ?></h4>
                            <p class="mb-0">Borçlu Tedarikçi</p>
                            <small>₺<?php echo number_format($stats['tedarikci_borc'], 2); ?></small>
                        </div>
                    </div>
                </div>
                <div class="col-md-3 mb-3">
                    <div class="card bg-secondary text-white">
                        <div class="card-body text-center">
                            <i class="fas fa-equals fa-2x mb-2"></i>
                            <h4><?php echo number_format($stats['sifir_bakiye']); ?></h4>
                            <p class="mb-0">Sıfır Bakiye</p>
                            <small>Toplam Borç: ₺<?php echo number_format($stats['toplam_borc'], 2); ?></small>
                        </div>
                    </div>
                </div>
            </div>

            
            <div class="card mb-4">
                <div class="card-header">
                    <h6 class="mb-0">
                        <i class="fas fa-filter me-2"></i>
                        Filtreler
                    </h6>
                </div>
                <div class="card-body">
                    <form method="GET" class="row g-3">
                        <div class="col-md-3">
                            <label class="form-label">Arama</label>
                            <input type="text" 
                                   class="form-control" 
                                   name="search" 
                                   placeholder="Cari adı, telefon, email..."
                                   value="<?php echo Security::escapeOutput($search); ?>">
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">Cari Tipi</label>
                            <select name="cari_tipi" class="form-select">
                                <option value="">Tümü</option>
                                <option value="musteri" <?php echo $cari_tipi === 'musteri' ? 'selected' : ''; ?>>Müşteri</option>
                                <option value="tedarikci" <?php echo $cari_tipi === 'tedarikci' ? 'selected' : ''; ?>>Tedarikçi</option>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">Bakiye Durumu</label>
                            <select name="bakiye_durumu" class="form-select">
                                <option value="">Tümü</option>
                                <option value="borclu" <?php echo $bakiye_durumu === 'borclu' ? 'selected' : ''; ?>>Borçlu</option>
                                <option value="alacakli" <?php echo $bakiye_durumu === 'alacakli' ? 'selected' : ''; ?>>Alacaklı</option>
                                <option value="sifir" <?php echo $bakiye_durumu === 'sifir' ? 'selected' : ''; ?>>Sıfır Bakiye</option>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">Başlangıç Tarihi</label>
                            <input type="date" 
                                   class="form-control" 
                                   name="tarih_baslangic"
                                   value="<?php echo Security::escapeOutput($tarih_baslangic); ?>">
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">Bitiş Tarihi</label>
                            <input type="date" 
                                   class="form-control" 
                                   name="tarih_bitis"
                                   value="<?php echo Security::escapeOutput($tarih_bitis); ?>">
                        </div>
                        <div class="col-md-1">
                            <label class="form-label">&nbsp;</label>
                            <button type="submit" class="btn btn-primary w-100">
                                <i class="fas fa-search"></i>
                            </button>
                        </div>
                    </form>
                </div>
            </div>

            
            <div class="card mb-4">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h6 class="mb-0">
                        <i class="fas fa-users me-2"></i>
                        Cari Listesi (<?php echo number_format($total_records); ?> kayıt)
                    </h6>
                </div>
                <div class="card-body p-0">
                    <?php if (empty($customers)): ?>
                        <div class="text-center py-5">
                            <i class="fas fa-users fa-3x text-muted mb-3"></i>
                            <h5 class="text-muted">Cari bulunamadı</h5>
                            <p class="text-muted">Arama kriterlerinize uygun cari bulunamadı.</p>
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead class="table-light">
                                    <tr>
                                        <th>Cari Adı</th>
                                        <th>Tipi</th>
                                        <th>İletişim</th>
                                        <th class="text-end">Bakiye</th>
                                        <th class="text-center">İşlem Sayısı</th>
                                        <th class="text-end">İşlem Hacmi</th>
                                        <th class="text-center">İşlemler</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($customers as $customer): ?>
                                        <tr>
                                            <td>
                                                <strong><?php echo Security::escapeOutput($customer['cari_adi']); ?></strong>
                                                <?php if ($customer['hareket_sayisi'] > 0): ?>
                                                    <br><small class="text-muted"><?php echo $customer['hareket_sayisi']; ?> hareket</small>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <span class="badge <?php echo $customer['cari_tipi'] === 'musteri' ? 'bg-primary' : 'bg-info'; ?>">
                                                    <?php echo $customer['cari_tipi'] === 'musteri' ? 'Müşteri' : 'Tedarikçi'; ?>
                                                </span>
                                            </td>
                                            <td>
                                                <?php if ($customer['telefon']): ?>
                                                    <i class="fas fa-phone me-1"></i>
                                                    <?php echo Security::escapeOutput($customer['telefon']); ?><br>
                                                <?php endif; ?>
                                                <?php if ($customer['email']): ?>
                                                    <i class="fas fa-envelope me-1"></i>
                                                    <?php echo Security::escapeOutput($customer['email']); ?>
                                                <?php endif; ?>
                                            </td>
                                            <td class="text-end">
                                                <?php
                                                $balance = (float)$customer['cari_bakiye'];
                                                // Müşteri için: pozitif bakiye = müşteri bize borçlu (kırmızı), negatif = müşteri bizden alacaklı (yeşil - olmamalı)
                                                // Tedarikçi için: pozitif bakiye = biz tedarikçiye borçlu (kırmızı), negatif = tedarikçi bize borçlu (yeşil - nadir durum)
                                                if ($customer['cari_tipi'] === 'musteri') {
                                                    $class = $balance > 0 ? 'balance-positive' : ($balance < 0 ? 'balance-negative' : 'balance-zero');
                                                    $label = $balance > 0 ? '(Bize Borçlu)' : ($balance < 0 ? '(Bizden Alacaklı - Hata!)' : '');
                                                } else { // tedarikci
                                                    $class = $balance > 0 ? 'balance-positive' : ($balance < 0 ? 'balance-negative' : 'balance-zero');
                                                    $label = $balance > 0 ? '(Tedarikçiye Borçluyuz)' : ($balance < 0 ? '(Tedarikçi Bize Borçlu)' : '');
                                                }
                                                ?>
                                                <span class="<?php echo $class; ?>">
                                                    ₺<?php echo number_format(abs($balance), 2); ?>
                                                    <?php if ($balance != 0): ?>
                                                        <small><?php echo $label; ?></small>
                                                    <?php endif; ?>
                                                </span>
                                            </td>
                                            <td class="text-center">
                                                <span class="badge bg-secondary"><?php echo $customer['toplam_satis']; ?></span>
                                            </td>
                                            <td class="text-end">
                                                <strong>₺<?php echo number_format($customer['toplam_alis_tutari'], 2); ?></strong>
                                            </td>
                                            <td class="text-center">
                                                <?php if ($customer['cari_tipi'] === 'musteri' && $customer['cari_bakiye'] > 0): ?>
                                                    <button class="btn btn-success btn-sm" 
                                                            onclick="showPaymentModal(<?php echo $customer['cari_id']; ?>, '<?php echo Security::escapeOutput($customer['cari_adi']); ?>', <?php echo $customer['cari_bakiye']; ?>, 'musteri')">
                                                        <i class="fas fa-money-bill-wave"></i>
                                                        Ödeme Al
                                                    </button>
                                                <?php elseif ($customer['cari_tipi'] === 'tedarikci' && $customer['cari_bakiye'] > 0): ?>
                                                    <button class="btn btn-warning btn-sm" 
                                                            onclick="showPaymentModal(<?php echo $customer['cari_id']; ?>, '<?php echo Security::escapeOutput($customer['cari_adi']); ?>', <?php echo $customer['cari_bakiye']; ?>, 'tedarikci')">
                                                        <i class="fas fa-hand-holding-usd"></i>
                                                        Ödeme Yap
                                                    </button>
                                                <?php endif; ?>
                                                <a href="customer_detail.php?id=<?php echo $customer['cari_id']; ?>" 
                                                   class="btn btn-outline-primary btn-sm">
                                                    <i class="fas fa-eye"></i>
                                                </a>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>

                        
                        <?php if ($total_pages > 1): ?>
                            <div class="card-footer">
                                <nav>
                                    <ul class="pagination mb-0">
                                        <?php if ($page > 1): ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page - 1])); ?>">
                                                    <i class="fas fa-chevron-left"></i>
                                                </a>
                                            </li>
                                        <?php endif; ?>
                                        
                                        <?php for ($i = max(1, $page - 2); $i <= min($total_pages, $page + 2); $i++): ?>
                                            <li class="page-item <?php echo $i === $page ? 'active' : ''; ?>">
                                                <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>">
                                                    <?php echo $i; ?>
                                                </a>
                                            </li>
                                        <?php endfor; ?>
                                        
                                        <?php if ($page < $total_pages): ?>
                                            <li class="page-item">
                                                <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page + 1])); ?>">
                                                    <i class="fas fa-chevron-right"></i>
                                                </a>
                                            </li>
                                        <?php endif; ?>
                                    </ul>
                                </nav>
                            </div>
                        <?php endif; ?>
                    <?php endif; ?>
                </div>
            </div>

            
            <?php if (!empty($recent_sales)): ?>
                <div class="card">
                    <div class="card-header">
                        <h6 class="mb-0">
                            <i class="fas fa-shopping-cart me-2"></i>
                            Son Satışlar
                        </h6>
                    </div>
                    <div class="card-body p-0">
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead class="table-light">
                                    <tr>
                                        <th>Satış No</th>
                                        <th>Müşteri</th>
                                        <th>Tarih</th>
                                        <th class="text-end">Toplam</th>
                                        <th class="text-end">Ödenen</th>
                                        <th class="text-end">Kalan</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($recent_sales as $sale): ?>
                                        <tr>
                                            <td><strong>#<?php echo $sale['satis_id']; ?></strong></td>
                                            <td><?php echo Security::escapeOutput($sale['cari_adi']); ?></td>
                                            <td><?php echo date('d.m.Y H:i', strtotime($sale['satis_tarihi'])); ?></td>
                                            <td class="text-end">₺<?php echo number_format($sale['toplam_tutar'], 2); ?></td>
                                            <td class="text-end">₺<?php echo number_format($sale['odenen_tutar'], 2); ?></td>
                                            <td class="text-end">
                                                <?php $kalan = $sale['toplam_tutar'] - $sale['odenen_tutar']; ?>
                                                <span class="<?php echo $kalan > 0 ? 'text-danger' : 'text-success'; ?>">
                                                    ₺<?php echo number_format($kalan, 2); ?>
                                                </span>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </div>

    
    <div class="modal fade" id="paymentModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <form method="POST">
                    <?php echo Security::getCSRFTokenInput(); ?>
                    <input type="hidden" name="odeme_yap" value="1">
                    <input type="hidden" name="cari_id" id="paymentCariId">
                    
                    <div class="modal-header">
                        <h5 class="modal-title">Ödeme Al</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="mb-3">
                            <strong id="paymentCariName"></strong>
                            <div class="text-muted">Mevcut Borç: ₺<span id="paymentCurrentBalance"></span></div>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Ödeme Tutarı (₺) *</label>
                            <input type="number" 
                                   name="odeme_tutari" 
                                   id="paymentAmount"
                                   class="form-control" 
                                   step="0.01" 
                                   min="0.01" 
                                   required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Açıklama</label>
                            <textarea name="aciklama" class="form-control" rows="2" placeholder="Ödeme açıklaması..."></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">İptal</button>
                        <button type="submit" class="btn btn-success">
                            <i class="fas fa-money-bill-wave me-2"></i>
                            Ödeme Al
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Sidebar toggle functionality is handled in header.php

        function showPaymentModal(cariId, cariName, currentBalance, cariType) {
            document.getElementById('paymentCariId').value = cariId;
            document.getElementById('paymentCariName').textContent = cariName;
            document.getElementById('paymentCurrentBalance').textContent = currentBalance.toFixed(2);
            document.getElementById('paymentAmount').max = currentBalance;
            document.getElementById('paymentAmount').value = currentBalance.toFixed(2);
            
            // Modal başlığını ve buton metnini cari tipine göre güncelle
            const modalTitle = document.querySelector('#paymentModal .modal-title');
            const submitButton = document.querySelector('#paymentModal button[type="submit"]');
            
            if (cariType === 'tedarikci') {
                modalTitle.textContent = 'Tedarikçi Ödemesi';
                submitButton.innerHTML = '<i class="fas fa-hand-holding-usd me-2"></i>Ödeme Yap';
                submitButton.className = 'btn btn-warning';
            } else {
                modalTitle.textContent = 'Müşteri Ödemesi';
                submitButton.innerHTML = '<i class="fas fa-money-bill-wave me-2"></i>Ödeme Al';
                submitButton.className = 'btn btn-success';
            }
            
            new bootstrap.Modal(document.getElementById('paymentModal')).show();
        }

        document.querySelector('#paymentModal form').addEventListener('submit', function(e) {
            const amount = parseFloat(document.getElementById('paymentAmount').value);
            const maxAmount = parseFloat(document.getElementById('paymentAmount').max);
            
            if (amount <= 0) {
                e.preventDefault();
                alert('Ödeme tutarı pozitif olmalıdır.');
                return false;
            }
            
            if (amount > maxAmount) {
                e.preventDefault();
                alert('Ödeme tutarı mevcut borçtan fazla olamaz.');
                return false;
            }
            
            if (!confirm('Ödemeyi kaydetmek istediğinizden emin misiniz?')) {
                e.preventDefault();
                return false;
            }
        });
    </script>
</main>

<?php include '../includes/footer.php'; ?>