<?php
require_once '../config/config.php';
require_once '../includes/security.php';
require_once '../includes/error_handler.php';
require_once '../config/database.php';

$error_message = '';
$success_message = '';

if (isset($_GET['message']) && $_GET['message'] === 'logout_success') {
    $success_message = 'Başarıyla çıkış yaptınız.';
}

if (Security::isLoggedIn()) {
    header('Location: dashboard.php');
    exit();
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        if (!isset($_POST['csrf_token']) || !Security::validateCSRFToken($_POST['csrf_token'])) {
            throw new Exception('Güvenlik hatası: Geçersiz form verisi.');
        }

        $client_ip = Security::getClientIP();
        
        if (!Security::checkLoginAttempts($client_ip)) {
            throw new Exception('Çok fazla başarısız deneme. 15 dakika sonra tekrar deneyin.');
        }

        $username = Security::sanitizeInput($_POST['username'] ?? '');
        $password = $_POST['password'] ?? '';

        if (empty($username) || empty($password)) {
            throw new Exception('Kullanıcı adı ve şifre gereklidir.');
        }

        $sql = "SELECT k.kullanici_id, k.kullanici_adi, k.sifre_hash, k.ad_soyad, k.rol_id, k.aktif, r.rol_adi 
                FROM kullanicilar k 
                LEFT JOIN roller r ON k.rol_id = r.rol_id 
                WHERE k.kullanici_adi = ? AND k.aktif = 1 
                LIMIT 1";
        $user = $database->fetch($sql, [$username]);

        if (!$user || !Security::verifyPassword($password, $user['sifre_hash'])) {
            Security::recordFailedLogin($client_ip);
            throw new Exception('Kullanıcı adı veya şifre hatalı.');
        }

        Security::resetLoginAttempts($client_ip);
        Security::loginUser($user['kullanici_id'], $user['kullanici_adi'], $user['rol_id']);
        
        header('Location: dashboard.php');
        exit();

    } catch (Exception $e) {
        $error_type = 'general';
        if (strpos($e->getMessage(), 'CSRF') !== false) {
            $error_type = 'csrf';
        } elseif (strpos($e->getMessage(), 'fazla') !== false) {
            $error_type = 'rate_limit';
        } elseif (strpos($e->getMessage(), 'Kullanıcı') !== false) {
            $error_type = 'validation';
        }
        
        ErrorHandler::logError($error_type, $e->getMessage(), [
            'ip' => $client_ip ?? Security::getClientIP(),
            'username' => $_POST['username'] ?? '',
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? ''
        ]);
        
        $error_message = ErrorHandler::getUserFriendlyMessage($error_type, $e->getMessage());
    }
}
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Giriş Yap - Depo Stok Takip Sistemi</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
        }
        .login-container {
            background: white;
            border-radius: 15px;
            box-shadow: 0 15px 35px rgba(0, 0, 0, 0.1);
            overflow: hidden;
        }
        .login-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 2rem;
            text-align: center;
        }
        .login-form {
            padding: 2rem;
        }
        .form-control {
            border-radius: 10px;
            border: 2px solid #e9ecef;
            padding: 12px 15px;
            transition: all 0.3s ease;
        }
        .form-control:focus {
            border-color: #667eea;
            box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.25);
        }
        .btn-login {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border: none;
            border-radius: 10px;
            padding: 12px;
            font-weight: 600;
            transition: all 0.3s ease;
        }
        .btn-login:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.4);
        }
        .input-group-text {
            background: #f8f9fa;
            border: 2px solid #e9ecef;
            border-right: none;
        }
        .input-group .form-control {
            border-left: none;
        }
        .alert {
            border-radius: 10px;
            border: none;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="row justify-content-center">
            <div class="col-md-6 col-lg-5">
                <div class="login-container">
                    <div class="login-header">
                        <i class="fas fa-warehouse fa-3x mb-3"></i>
                        <h3 class="mb-0">Depo Stok Takip</h3>
                        <p class="mb-0">Sisteme Giriş Yapın</p>
                    </div>
                    
                    <div class="login-form">
                        <?php if ($error_message): ?>
                            <div class="alert alert-danger" role="alert">
                                <i class="fas fa-exclamation-triangle me-2"></i>
                                <?php echo Security::escapeOutput($error_message); ?>
                            </div>
                        <?php endif; ?>

                        <?php if ($success_message): ?>
                            <div class="alert alert-success" role="alert">
                                <i class="fas fa-check-circle me-2"></i>
                                <?php echo Security::escapeOutput($success_message); ?>
                            </div>
                        <?php endif; ?>

                        <form method="POST" action="" novalidate>
                            <?php echo Security::getCSRFTokenInput(); ?>
                            
                            <div class="mb-3">
                                <label for="username" class="form-label">Kullanıcı Adı</label>
                                <div class="input-group">
                                    <span class="input-group-text">
                                        <i class="fas fa-user"></i>
                                    </span>
                                    <input type="text" 
                                           class="form-control" 
                                           id="username" 
                                           name="username" 
                                           placeholder="Kullanıcı adınızı girin"
                                           value="<?php echo isset($_POST['username']) ? Security::escapeOutput($_POST['username']) : ''; ?>"
                                           required 
                                           autocomplete="username">
                                </div>
                            </div>

                            <div class="mb-4">
                                <label for="password" class="form-label">Şifre</label>
                                <div class="input-group">
                                    <span class="input-group-text">
                                        <i class="fas fa-lock"></i>
                                    </span>
                                    <input type="password" 
                                           class="form-control" 
                                           id="password" 
                                           name="password" 
                                           placeholder="Şifrenizi girin"
                                           required 
                                           autocomplete="current-password">
                                    <button class="btn btn-outline-secondary" 
                                            type="button" 
                                            id="togglePassword">
                                        <i class="fas fa-eye"></i>
                                    </button>
                                </div>
                            </div>

                            <div class="d-grid">
                                <button type="submit" class="btn btn-primary btn-login">
                                    <i class="fas fa-sign-in-alt me-2"></i>
                                    Giriş Yap
                                </button>
                            </div>
                        </form>

                        <div class="text-center mt-4">
                            <small class="text-muted">
                                <i class="fas fa-shield-alt me-1"></i>
                                Güvenli bağlantı ile korunmaktadır
                            </small>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        document.getElementById('togglePassword').addEventListener('click', function() {
            const password = document.getElementById('password');
            const icon = this.querySelector('i');
            
            if (password.type === 'password') {
                password.type = 'text';
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
            } else {
                password.type = 'password';
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
            }
        });

        document.querySelector('form').addEventListener('submit', function(e) {
            const username = document.getElementById('username').value.trim();
            const password = document.getElementById('password').value;

            if (!username || !password) {
                e.preventDefault();
                alert('Lütfen tüm alanları doldurun.');
                return false;
            }

            if (username.length < 3) {
                e.preventDefault();
                alert('Kullanıcı adı en az 3 karakter olmalıdır.');
                return false;
            }
        });

        document.getElementById('username').focus();
    </script>
</body>
</html>

