<?php
session_start();
require_once '../config/config.php';
require_once '../includes/security.php';
require_once '../includes/authorization.php';
require_once '../includes/error_handler.php';

Authorization::requireAdmin();

$message = '';
$action = Security::sanitizeInput($_POST['action'] ?? $_GET['action'] ?? '');

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    Security::validateCSRFToken($_POST['csrf_token'] ?? '');
    
    try {
        switch ($action) {
            case 'cleanup_logs':
                $days = (int)($_POST['days_to_keep'] ?? 30);
                ErrorHandler::cleanupLogs($days);
                $message = "Son $days günden eski loglar temizlendi.";
                break;
                
            case 'clear_all_logs':
                $log_file = __DIR__ . '/../logs/error.log';
                if (file_exists($log_file)) {
                    file_put_contents($log_file, '');
                    $message = 'Tüm hata logları temizlendi.';
                } else {
                    $message = 'Log dosyası bulunamadı.';
                }
                break;
        }
    } catch (Exception $e) {
        $message = 'İşlem sırasında hata oluştu: ' . $e->getMessage();
    }
}

$error_stats = ErrorHandler::getErrorStats(7); // Son 7 gün
$error_stats_30 = ErrorHandler::getErrorStats(30); // Son 30 gün

$csrf_token = Security::generateCSRFToken();

include '../includes/header.php';
?>
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>

<?php include '../includes/sidebar.php'; ?>

<div class="main-content">
    <div class="container-fluid">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2><i class="fas fa-bug me-2"></i>Hata Yönetimi</h2>
                    <a href="dashboard.php" class="btn btn-secondary">
                        <i class="fas fa-arrow-left me-1"></i>Dashboard'a Dön
                    </a>
                </div>

                <?php if ($message): ?>
                    <div class="alert alert-info alert-dismissible fade show" role="alert">
                        <i class="fas fa-info-circle me-2"></i>
                        <?= htmlspecialchars($message) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                
                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="card text-white bg-danger">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <h4 class="card-title"><?= $error_stats['total_errors'] ?></h4>
                                        <p class="card-text">Son 7 Gün</p>
                                    </div>
                                    <div class="align-self-center">
                                        <i class="fas fa-exclamation-triangle fa-2x"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card text-white bg-warning">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <h4 class="card-title"><?= $error_stats_30['total_errors'] ?></h4>
                                        <p class="card-text">Son 30 Gün</p>
                                    </div>
                                    <div class="align-self-center">
                                        <i class="fas fa-chart-line fa-2x"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card text-white bg-info">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <h4 class="card-title"><?= count($error_stats['by_type']) ?></h4>
                                        <p class="card-text">Hata Türü</p>
                                    </div>
                                    <div class="align-self-center">
                                        <i class="fas fa-tags fa-2x"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card text-white bg-success">
                            <div class="card-body">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <h4 class="card-title"><?= file_exists(__DIR__ . '/../logs/error.log') ? 'Aktif' : 'Pasif' ?></h4>
                                        <p class="card-text">Log Durumu</p>
                                    </div>
                                    <div class="align-self-center">
                                        <i class="fas fa-file-alt fa-2x"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="row">
                    
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0"><i class="fas fa-chart-pie me-2"></i>Hata Türleri (Son 7 Gün)</h5>
                            </div>
                            <div class="card-body">
                                <canvas id="errorTypesChart" width="400" height="200"></canvas>
                            </div>
                        </div>
                    </div>
                    
                    
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0"><i class="fas fa-chart-line me-2"></i>Günlük Hata Sayısı</h5>
                            </div>
                            <div class="card-body">
                                <canvas id="dailyErrorsChart" width="400" height="200"></canvas>
                            </div>
                        </div>
                    </div>
                </div>

                
                <div class="card mt-4">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="fas fa-list me-2"></i>En Sık Görülen Hatalar (Son 7 Gün)</h5>
                    </div>
                    <div class="card-body">
                        <?php if (!empty($error_stats['top_errors'])): ?>
                            <div class="table-responsive">
                                <table class="table table-striped">
                                    <thead>
                                        <tr>
                                            <th>Hata Mesajı</th>
                                            <th>Görülme Sayısı</th>
                                            <th>Yüzde</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($error_stats['top_errors'] as $error => $count): ?>
                                            <tr>
                                                <td><?= htmlspecialchars(substr($error, 0, 100)) ?><?= strlen($error) > 100 ? '...' : '' ?></td>
                                                <td><span class="badge bg-danger"><?= $count ?></span></td>
                                                <td><?= round(($count / $error_stats['total_errors']) * 100, 1) ?>%</td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php else: ?>
                            <div class="text-center text-muted">
                                <i class="fas fa-check-circle fa-3x mb-3"></i>
                                <p>Son 7 günde hata kaydı bulunmuyor.</p>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                
                <div class="card mt-4">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="fas fa-cogs me-2"></i>Log Yönetimi</h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6">
                                <form method="POST" class="mb-3">
                                    <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">
                                    <input type="hidden" name="action" value="cleanup_logs">
                                    
                                    <div class="mb-3">
                                        <label for="days_to_keep" class="form-label">Kaç günlük log tutulsun?</label>
                                        <select class="form-select" id="days_to_keep" name="days_to_keep">
                                            <option value="7">7 Gün</option>
                                            <option value="15">15 Gün</option>
                                            <option value="30" selected>30 Gün</option>
                                            <option value="60">60 Gün</option>
                                            <option value="90">90 Gün</option>
                                        </select>
                                    </div>
                                    
                                    <button type="submit" class="btn btn-warning">
                                        <i class="fas fa-broom me-1"></i>Eski Logları Temizle
                                    </button>
                                </form>
                            </div>
                            
                            <div class="col-md-6">
                                <form method="POST" onsubmit="return confirm('Tüm hata logları silinecek. Emin misiniz?')">
                                    <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">
                                    <input type="hidden" name="action" value="clear_all_logs">
                                    
                                    <div class="mb-3">
                                        <label class="form-label">Tüm Logları Temizle</label>
                                        <p class="text-muted small">Bu işlem geri alınamaz!</p>
                                    </div>
                                    
                                    <button type="submit" class="btn btn-danger">
                                        <i class="fas fa-trash me-1"></i>Tüm Logları Sil
                                    </button>
                                </form>
                            </div>
                        </div>
                        
                        <hr>
                        
                        <div class="row">
                            <div class="col-md-12">
                                <h6>Log Dosya Bilgileri:</h6>
                                <?php 
                                $log_file = __DIR__ . '/../logs/error.log';
                                if (file_exists($log_file)): 
                                    $file_size = filesize($log_file);
                                    $file_date = date('d.m.Y H:i:s', filemtime($log_file));
                                ?>
                                    <ul class="list-unstyled">
                                        <li><strong>Dosya Boyutu:</strong> <?= number_format($file_size / 1024, 2) ?> KB</li>
                                        <li><strong>Son Güncelleme:</strong> <?= $file_date ?></li>
                                        <li><strong>Dosya Yolu:</strong> <code><?= $log_file ?></code></li>
                                    </ul>
                                <?php else: ?>
                                    <p class="text-muted">Log dosyası henüz oluşturulmamış.</p>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
    </div>
</div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        const errorTypesCtx = document.getElementById('errorTypesChart').getContext('2d');
        const errorTypesData = <?= json_encode($error_stats['by_type']) ?>;
        
        new Chart(errorTypesCtx, {
            type: 'doughnut',
            data: {
                labels: Object.keys(errorTypesData),
                datasets: [{
                    data: Object.values(errorTypesData),
                    backgroundColor: [
                        '#FF6384', '#36A2EB', '#FFCE56', '#4BC0C0', 
                        '#9966FF', '#FF9F40', '#FF6384', '#C9CBCF'
                    ]
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        });

        const dailyErrorsCtx = document.getElementById('dailyErrorsChart').getContext('2d');
        const dailyErrorsData = <?= json_encode($error_stats['by_date']) ?>;
        
        const last7Days = [];
        const errorCounts = [];
        for (let i = 6; i >= 0; i--) {
            const date = new Date();
            date.setDate(date.getDate() - i);
            const dateStr = date.toISOString().split('T')[0];
            last7Days.push(date.toLocaleDateString('tr-TR', { month: 'short', day: 'numeric' }));
            errorCounts.push(dailyErrorsData[dateStr] || 0);
        }
        
        new Chart(dailyErrorsCtx, {
            type: 'line',
            data: {
                labels: last7Days,
                datasets: [{
                    label: 'Hata Sayısı',
                    data: errorCounts,
                    borderColor: '#FF6384',
                    backgroundColor: 'rgba(255, 99, 132, 0.1)',
                    tension: 0.4,
                    fill: true
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            stepSize: 1
                        }
                    }
                },
                plugins: {
                    legend: {
                        display: false
                    }
                }
            }
        });
    </script>
</body>
</html>