<?php
session_start();
require_once '../includes/security.php';
require_once '../includes/authorization.php';
require_once '../includes/balance_manager.php';

Authorization::requireRole('admin');

$balance_manager = new BalanceManager();
$message = '';
$report = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    Security::validateCSRFToken($_POST['csrf_token'] ?? '');
    
    $action = Security::sanitizeInput($_POST['action'] ?? '');
    
    try {
        switch ($action) {
            case 'check_consistency':
                $inconsistencies = $balance_manager->checkBalanceConsistency();
                $report = [
                    'type' => 'consistency_check',
                    'inconsistencies' => $inconsistencies,
                    'count' => count($inconsistencies)
                ];
                $message = count($inconsistencies) > 0 
                    ? 'Tutarsızlık bulunan ' . count($inconsistencies) . ' cari hesap var.'
                    : 'Tüm bakiyeler tutarlı.';
                break;
                
            case 'update_all_balances':
                $result = $balance_manager->updateAllBalances();
                $report = [
                    'type' => 'balance_update',
                    'result' => $result
                ];
                $message = "Bakiye güncelleme tamamlandı. Güncellenen: {$result['updated']}, Hata: {$result['errors']}, Toplam: {$result['total']}";
                break;
                
            case 'full_optimization':
                $report = $balance_manager->runOptimization();
                $report['type'] = 'full_optimization';
                $message = 'Tam optimizasyon tamamlandı.';
                break;
                
            case 'balance_summary':
                $summary = $balance_manager->getBalanceSummary();
                $report = [
                    'type' => 'summary',
                    'data' => $summary
                ];
                $message = 'Bakiye özeti hazırlandı.';
                break;
                
            case 'supplier_debt_check':
                $supplier_debts = $balance_manager->checkSupplierDebtMechanism();
                $report = [
                    'type' => 'supplier_debt_check',
                    'data' => $supplier_debts
                ];
                $message = 'Tedarikçi borçlanma mekanizması kontrolü tamamlandı.';
                break;
        }
    } catch (Exception $e) {
        $message = 'Hata: ' . $e->getMessage();
    }
}

$csrf_token = Security::generateCSRFToken();

include '../includes/header.php';
?>

<?php include '../includes/sidebar.php'; ?>

<div class="main-content">
    <div class="container-fluid">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2><i class="fas fa-database me-2"></i>Veritabanı Optimizasyonu</h2>
                    <a href="dashboard.php" class="btn btn-secondary">
                        <i class="fas fa-arrow-left me-1"></i>Dashboard'a Dön
                    </a>
                </div>

                <?php if ($message): ?>
                    <div class="alert alert-info alert-dismissible fade show" role="alert">
                        <i class="fas fa-info-circle me-2"></i><?= htmlspecialchars($message) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                
                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="card h-100">
                            <div class="card-body text-center">
                                <i class="fas fa-search fa-3x text-primary mb-3"></i>
                                <h5 class="card-title">Tutarlılık Kontrolü</h5>
                                <p class="card-text">Bakiye tutarlılığını kontrol eder</p>
                                <form method="POST" class="d-inline">
                                    <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">
                                    <input type="hidden" name="action" value="check_consistency">
                                    <button type="submit" class="btn btn-primary">
                                        <i class="fas fa-search me-1"></i>Kontrol Et
                                    </button>
                                </form>
                            </div>
                        </div>
                        </div>
                    
                    <div class="col-md-3">
                        <div class="card h-100">
                            <div class="card-body text-center">
                                <i class="fas fa-sync fa-3x text-warning mb-3"></i>
                                <h5 class="card-title">Bakiye Güncelleme</h5>
                                <p class="card-text">Tüm bakiyeleri yeniden hesaplar</p>
                                <form method="POST" class="d-inline">
                                    <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">
                                    <input type="hidden" name="action" value="update_all_balances">
                                    <button type="submit" class="btn btn-warning" onclick="return confirm('Tüm bakiyeler yeniden hesaplanacak. Devam etmek istiyor musunuz?')">
                                        <i class="fas fa-sync me-1"></i>Güncelle
                                    </button>
                                </form>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-3">
                        <div class="card h-100">
                            <div class="card-body text-center">
                                <i class="fas fa-cogs fa-3x text-success mb-3"></i>
                                <h5 class="card-title">Tam Optimizasyon</h5>
                                <p class="card-text">Kapsamlı optimizasyon yapar</p>
                                <form method="POST" class="d-inline">
                                    <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">
                                    <input type="hidden" name="action" value="full_optimization">
                                    <button type="submit" class="btn btn-success" onclick="return confirm('Tam optimizasyon çalıştırılacak. Bu işlem biraz zaman alabilir. Devam etmek istiyor musunuz?')">
                                        <i class="fas fa-cogs me-1"></i>Optimizasyon
                                    </button>
                                </form>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-3">
                        <div class="card h-100">
                            <div class="card-body text-center">
                                <i class="fas fa-chart-pie fa-3x text-info mb-3"></i>
                                <h5 class="card-title">Bakiye Özeti</h5>
                                <p class="card-text">Genel bakiye raporunu gösterir</p>
                                <form method="POST" class="d-inline">
                                    <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">
                                    <input type="hidden" name="action" value="balance_summary">
                                    <button type="submit" class="btn btn-info">
                                        <i class="fas fa-chart-pie me-1"></i>Özet
                                    </button>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- İkinci satır - Tedarikçi Borç Kontrolü -->
                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="card h-100">
                            <div class="card-body text-center">
                                <i class="fas fa-truck fa-3x text-danger mb-3"></i>
                                <h5 class="card-title">Tedarikçi Borç Kontrolü</h5>
                                <p class="card-text">Tedarikçi borçlanma mekanizmasını kontrol eder</p>
                                <form method="POST" class="d-inline">
                                    <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">
                                    <input type="hidden" name="action" value="supplier_debt_check">
                                    <button type="submit" class="btn btn-danger">
                                        <i class="fas fa-truck me-1"></i>Kontrol Et
                                    </button>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>

                
                <?php if ($report): ?>
                    <div class="card">
                        <div class="card-header">
                            <h5 class="mb-0">
                                <i class="fas fa-file-alt me-2"></i>
                                <?php
                                switch ($report['type']) {
                                    case 'consistency_check':
                                        echo 'Tutarlılık Kontrolü Sonucu';
                                        break;
                                    case 'balance_update':
                                        echo 'Bakiye Güncelleme Sonucu';
                                        break;
                                    case 'full_optimization':
                                        echo 'Tam Optimizasyon Raporu';
                                        break;
                                    case 'summary':
                                        echo 'Bakiye Özeti';
                                        break;
                                    case 'supplier_debt_check':
                                        echo 'Tedarikçi Borçlanma Mekanizması Kontrolü';
                                        break;
                                }
                                ?>
                            </h5>
                        </div>
                        <div class="card-body">
                            <?php if ($report['type'] === 'consistency_check'): ?>
                                <?php if ($report['count'] > 0): ?>
                                    <div class="alert alert-warning">
                                        <i class="fas fa-exclamation-triangle me-2"></i>
                                        <?= $report['count'] ?> cari hesapta bakiye tutarsızlığı bulundu.
                                    </div>
                                    <div class="table-responsive">
                                        <table class="table table-striped">
                                            <thead>
                                                <tr>
                                                    <th>Cari ID</th>
                                                    <th>Firma Adı</th>
                                                    <th>Kayıtlı Bakiye</th>
                                                    <th>Hesaplanan Bakiye</th>
                                                    <th>Fark</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach ($report['inconsistencies'] as $item): ?>
                                                    <tr>
                                                        <td><?= $item['cari_id'] ?></td>
                                                        <td><?= htmlspecialchars($item['firma_adi']) ?></td>
                                                        <td><?= number_format($item['stored_balance'], 2) ?> ₺</td>
                                                        <td><?= number_format($item['calculated_balance'], 2) ?> ₺</td>
                                                        <td class="text-danger">
                                                            <?= number_format($item['stored_balance'] - $item['calculated_balance'], 2) ?> ₺
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                <?php else: ?>
                                    <div class="alert alert-success">
                                        <i class="fas fa-check-circle me-2"></i>
                                        Tüm bakiyeler tutarlı. Herhangi bir sorun bulunamadı.
                                    </div>
                                <?php endif; ?>
                                
                            <?php elseif ($report['type'] === 'balance_update'): ?>
                                <div class="row">
                                    <div class="col-md-4">
                                        <div class="card bg-success text-white">
                                            <div class="card-body text-center">
                                                <h3><?= $report['result']['updated'] ?></h3>
                                                <p>Güncellenen</p>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="card bg-danger text-white">
                                            <div class="card-body text-center">
                                                <h3><?= $report['result']['errors'] ?></h3>
                                                <p>Hata</p>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="card bg-info text-white">
                                            <div class="card-body text-center">
                                                <h3><?= $report['result']['total'] ?></h3>
                                                <p>Toplam</p>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                            <?php elseif ($report['type'] === 'full_optimization'): ?>
                                <div class="row">
                                    <div class="col-md-6">
                                        <h6>Optimizasyon Öncesi</h6>
                                        <ul class="list-group">
                                            <li class="list-group-item d-flex justify-content-between">
                                                <span>Tutarsızlık Bulunan Hesap:</span>
                                                <span class="badge bg-warning"><?= $report['inconsistencies_found'] ?></span>
                                            </li>
                                        </ul>
                                    </div>
                                    <div class="col-md-6">
                                        <h6>Optimizasyon Sonrası</h6>
                                        <ul class="list-group">
                                            <li class="list-group-item d-flex justify-content-between">
                                                <span>Kalan Tutarsızlık:</span>
                                                <span class="badge bg-<?= $report['final_inconsistencies'] > 0 ? 'danger' : 'success' ?>">
                                                    <?= $report['final_inconsistencies'] ?>
                                                </span>
                                            </li>
                                            <li class="list-group-item d-flex justify-content-between">
                                                <span>Güncellenen Hesap:</span>
                                                <span class="badge bg-success"><?= $report['balance_update']['updated'] ?></span>
                                            </li>
                                        </ul>
                                    </div>
                                </div>
                                
                            <?php elseif ($report['type'] === 'summary'): ?>
                                <div class="row">
                                    <div class="col-md-4">
                                        <div class="card border-primary">
                                            <div class="card-body text-center">
                                                <h4 class="text-primary"><?= $report['data']['toplam_cari'] ?></h4>
                                                <p>Toplam Cari</p>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="card border-danger">
                                            <div class="card-body text-center">
                                                <h4 class="text-danger"><?= number_format($report['data']['toplam_borc'], 2) ?> ₺</h4>
                                                <p>Toplam Borç (<?= $report['data']['borclu_cari'] ?> cari)</p>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="card border-success">
                                            <div class="card-body text-center">
                                                <h4 class="text-success"><?= number_format($report['data']['toplam_alacak'], 2) ?> ₺</h4>
                                                <p>Toplam Alacak (<?= $report['data']['alacakli_cari'] ?> cari)</p>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <div class="row mt-3">
                                    <div class="col-md-6">
                                        <div class="card border-info">
                                            <div class="card-body text-center">
                                                <h4 class="text-info"><?= $report['data']['sifir_bakiye'] ?></h4>
                                                <p>Sıfır Bakiyeli Cari</p>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="card border-dark">
                                            <div class="card-body text-center">
                                                <h4 class="<?= $report['data']['net_bakiye'] >= 0 ? 'text-success' : 'text-danger' ?>">
                                                    <?= number_format($report['data']['net_bakiye'], 2) ?> ₺
                                                </h4>
                                                <p>Net Bakiye</p>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                            <?php elseif ($report['type'] === 'supplier_debt_check'): ?>
                                <div class="row">
                                    <div class="col-md-4">
                                        <div class="card border-danger">
                                            <div class="card-body text-center">
                                                <h4 class="text-danger"><?= $report['data']['toplam_tedarikci'] ?></h4>
                                                <p>Toplam Tedarikçi</p>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="card border-warning">
                                            <div class="card-body text-center">
                                                <h4 class="text-warning"><?= $report['data']['borclu_tedarikci'] ?></h4>
                                                <p>Borçlu Tedarikçi</p>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="card border-success">
                                            <div class="card-body text-center">
                                                <h4 class="text-success"><?= number_format($report['data']['toplam_tedarikci_borcu'], 2) ?> ₺</h4>
                                                <p>Toplam Tedarikçi Borcu</p>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <?php if (!empty($report['data']['problematic_suppliers'])): ?>
                                    <div class="mt-4">
                                        <h6>Problemli Tedarikçiler:</h6>
                                        <div class="table-responsive">
                                            <table class="table table-striped">
                                                <thead>
                                                    <tr>
                                                        <th>Tedarikçi ID</th>
                                                        <th>Firma Adı</th>
                                                        <th>Bakiye</th>
                                                        <th>Problem</th>
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                    <?php foreach ($report['data']['problematic_suppliers'] as $supplier): ?>
                                                        <tr>
                                                            <td><?= $supplier['cari_id'] ?></td>
                                                            <td><?= htmlspecialchars($supplier['firma_adi']) ?></td>
                                                            <td class="<?= $supplier['bakiye'] < 0 ? 'text-danger' : 'text-success' ?>">
                                                                <?= number_format($supplier['bakiye'], 2) ?> ₺
                                                            </td>
                                                            <td>
                                                                <span class="badge bg-warning">
                                                                    <?= htmlspecialchars($supplier['problem']) ?>
                                                                </span>
                                                            </td>
                                                        </tr>
                                                    <?php endforeach; ?>
                                                </tbody>
                                            </table>
                                        </div>
                                    </div>
                                <?php else: ?>
                                    <div class="alert alert-success mt-4">
                                        <i class="fas fa-check-circle me-2"></i>
                                        Tedarikçi borçlanma mekanizmasında herhangi bir problem bulunamadı.
                                    </div>
                                <?php endif; ?>
                                
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endif; ?>
    </div>
</div>

<?php include '../includes/footer.php'; ?>