<?php

class PerformanceMonitor {
    private $db;
    private $start_time;
    private $start_memory;
    
    public function __construct($database) {
        $this->db = $database;
        $this->start_time = microtime(true);
        $this->start_memory = memory_get_usage();
    }
    
    public function getDatabaseStats() {
        try {
            $stats = [];
            
            $query = "
                SELECT 
                    table_name,
                    ROUND(((data_length + index_length) / 1024 / 1024), 2) AS size_mb,
                    table_rows,
                    ROUND((data_length / 1024 / 1024), 2) AS data_size_mb,
                    ROUND((index_length / 1024 / 1024), 2) AS index_size_mb
                FROM information_schema.tables 
                WHERE table_schema = DATABASE()
                ORDER BY (data_length + index_length) DESC
            ";
            
            $stmt = $this->db->getConnection()->prepare($query);
            $stmt->execute();
            $stats['table_sizes'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $stmt = $this->db->getConnection()->prepare("SHOW VARIABLES LIKE 'slow_query_log'");
            $stmt->execute();
            $slow_log = $stmt->fetch(PDO::FETCH_ASSOC);
            $stats['slow_query_log'] = $slow_log['Value'] ?? 'OFF';
            
            $stmt = $this->db->getConnection()->prepare("SHOW STATUS LIKE 'Threads_connected'");
            $stmt->execute();
            $connections = $stmt->fetch(PDO::FETCH_ASSOC);
            $stats['active_connections'] = $connections['Value'] ?? 0;
            
            $stmt = $this->db->getConnection()->prepare("SHOW STATUS LIKE 'Qcache%'");
            $stmt->execute();
            $cache_stats = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $stats['query_cache'] = $cache_stats;
            
            return $stats;
            
        } catch (Exception $e) {
            error_log("Veritabanı performans istatistikleri alınırken hata: " . $e->getMessage());
            return [];
        }
    }
    
    public function getSystemStats() {
        $stats = [];
        
        $stats['memory_usage'] = [
            'current' => round(memory_get_usage() / 1024 / 1024, 2),
            'peak' => round(memory_get_peak_usage() / 1024 / 1024, 2),
            'limit' => ini_get('memory_limit')
        ];
        
        $stats['execution_time'] = round(microtime(true) - $this->start_time, 4);
        
        $stats['php_config'] = [
            'version' => PHP_VERSION,
            'max_execution_time' => ini_get('max_execution_time'),
            'upload_max_filesize' => ini_get('upload_max_filesize'),
            'post_max_size' => ini_get('post_max_size')
        ];
        
        if (function_exists('disk_free_space')) {
            $bytes = disk_free_space(".");
            $stats['disk_free'] = round($bytes / 1024 / 1024 / 1024, 2) . ' GB';
        }
        
        return $stats;
    }
    
    public function getSlowQueries() {
        try {
            $this->createPerformanceLogTable();
            
            $query = "
                SELECT 
                    operation_type,
                    table_name,
                    execution_time,
                    memory_usage,
                    created_at,
                    details
                FROM performance_log 
                WHERE created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
                AND execution_time > 1.0
                ORDER BY execution_time DESC
                LIMIT 20
            ";
            
            $stmt = $this->db->getConnection()->prepare($query);
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
            
        } catch (Exception $e) {
            error_log("Yavaş sorgular alınırken hata: " . $e->getMessage());
            return [];
        }
    }
    
    private function createPerformanceLogTable() {
        try {
            $query = "
                CREATE TABLE IF NOT EXISTS performance_log (
                    log_id INT AUTO_INCREMENT PRIMARY KEY,
                    operation_type VARCHAR(50) NOT NULL,
                    table_name VARCHAR(100),
                    execution_time DECIMAL(10,4) NOT NULL,
                    memory_usage DECIMAL(10,2),
                    query_hash VARCHAR(32),
                    details TEXT,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    INDEX idx_created_at (created_at),
                    INDEX idx_execution_time (execution_time),
                    INDEX idx_operation_type (operation_type)
                )
            ";
            
            $this->db->getConnection()->exec($query);
            
        } catch (Exception $e) {
            error_log("Performans log tablosu oluşturulurken hata: " . $e->getMessage());
        }
    }
    
    public function logOperation($operation_type, $table_name = null, $details = null) {
        try {
            $execution_time = microtime(true) - $this->start_time;
            $memory_usage = round((memory_get_usage() - $this->start_memory) / 1024 / 1024, 2);
            
            $this->createPerformanceLogTable();
            
            $query = "
                INSERT INTO performance_log 
                (operation_type, table_name, execution_time, memory_usage, details)
                VALUES (?, ?, ?, ?, ?)
            ";
            
            $stmt = $this->db->getConnection()->prepare($query);
            $stmt->execute([
                $operation_type,
                $table_name,
                $execution_time,
                $memory_usage,
                $details
            ]);
            
        } catch (Exception $e) {
            error_log("Performans logu kaydedilirken hata: " . $e->getMessage());
        }
    }
    
    public function getPerformanceRecommendations() {
        $recommendations = [];
        $db_stats = $this->getDatabaseStats();
        $system_stats = $this->getSystemStats();
        
        if (!empty($db_stats['table_sizes'])) {
            foreach ($db_stats['table_sizes'] as $table) {
                if ($table['size_mb'] > 100) {
                    $recommendations[] = [
                        'type' => 'warning',
                        'title' => 'Büyük Tablo',
                        'message' => "'{$table['table_name']}' tablosu {$table['size_mb']} MB boyutunda. Arşivleme düşünün.",
                        'action' => 'Eski kayıtları arşivleyin veya silin'
                    ];
                }
            }
        }
        
        if ($system_stats['memory_usage']['current'] > 64) {
            $recommendations[] = [
                'type' => 'warning',
                'title' => 'Yüksek Bellek Kullanımı',
                'message' => "Mevcut bellek kullanımı: {$system_stats['memory_usage']['current']} MB",
                'action' => 'Kod optimizasyonu yapın veya bellek limitini artırın'
            ];
        }
        
        if ($system_stats['execution_time'] > 5) {
            $recommendations[] = [
                'type' => 'error',
                'title' => 'Yavaş Sayfa Yükleme',
                'message' => "Sayfa yükleme süresi: {$system_stats['execution_time']} saniye",
                'action' => 'Veritabanı sorgularını optimize edin'
            ];
        }
        
        if (!empty($db_stats['query_cache'])) {
            $cache_hit_rate = 0;
            $cache_hits = 0;
            $cache_inserts = 0;
            
            foreach ($db_stats['query_cache'] as $stat) {
                if ($stat['Variable_name'] == 'Qcache_hits') {
                    $cache_hits = $stat['Value'];
                } elseif ($stat['Variable_name'] == 'Qcache_inserts') {
                    $cache_inserts = $stat['Value'];
                }
            }
            
            if ($cache_hits + $cache_inserts > 0) {
                $cache_hit_rate = ($cache_hits / ($cache_hits + $cache_inserts)) * 100;
                
                if ($cache_hit_rate < 80) {
                    $recommendations[] = [
                        'type' => 'info',
                        'title' => 'Düşük Cache Hit Rate',
                        'message' => "Query cache hit rate: " . round($cache_hit_rate, 2) . "%",
                        'action' => 'Query cache boyutunu artırın'
                    ];
                }
            }
        }
        
        return $recommendations;
    }
    
    public function generateReport() {
        return [
            'database_stats' => $this->getDatabaseStats(),
            'system_stats' => $this->getSystemStats(),
            'slow_queries' => $this->getSlowQueries(),
            'recommendations' => $this->getPerformanceRecommendations(),
            'generated_at' => date('Y-m-d H:i:s')
        ];
    }
    
    public function cleanupOldLogs($days = 7) {
        try {
            $query = "DELETE FROM performance_log WHERE created_at < DATE_SUB(NOW(), INTERVAL ? DAY)";
            $stmt = $this->db->getConnection()->prepare($query);
            $stmt->execute([$days]);
            
            return $stmt->rowCount();
            
        } catch (Exception $e) {
            error_log("Eski performans logları temizlenirken hata: " . $e->getMessage());
            return 0;
        }
    }
}
?>