<?php

class ErrorHandler {
    
    private static $error_log_file = __DIR__ . '/../logs/error.log';
    private static $user_friendly_messages = [
        'database' => 'Veritabanı bağlantısında bir sorun oluştu. Lütfen daha sonra tekrar deneyin.',
        'validation' => 'Girdiğiniz bilgilerde hata var. Lütfen kontrol edip tekrar deneyin.',
        'authorization' => 'Bu işlemi gerçekleştirmek için yetkiniz bulunmuyor.',
        'file_upload' => 'Dosya yükleme sırasında bir hata oluştu. Lütfen dosyanızı kontrol edin.',
        'session' => 'Oturumunuz sonlanmış. Lütfen tekrar giriş yapın.',
        'csrf' => 'Güvenlik hatası tespit edildi. Sayfayı yenileyin ve tekrar deneyin.',
        'rate_limit' => 'Çok fazla deneme yaptınız. Lütfen bir süre bekleyin.',
        'not_found' => 'Aradığınız kayıt bulunamadı.',
        'duplicate' => 'Bu bilgi zaten sistemde kayıtlı.',
        'insufficient_stock' => 'Yetersiz stok miktarı.',
        'invalid_input' => 'Geçersiz veri girişi.',
        'general' => 'Beklenmeyen bir hata oluştu. Lütfen daha sonra tekrar deneyin.'
    ];
    
    public static function getUserFriendlyMessage($error_type, $custom_message = null) {
        if ($custom_message) {
            return $custom_message;
        }
        
        return self::$user_friendly_messages[$error_type] ?? self::$user_friendly_messages['general'];
    }
    
    public static function handleError($exception, $error_type = 'general', $context = []) {
        self::logError($exception, $error_type, $context);
        
        return self::getUserFriendlyMessage($error_type);
    }
    
    public static function logError($exception, $error_type = 'general', $context = []) {
        if (is_string($exception)) {
            $log_entry = [
                'timestamp' => date('Y-m-d H:i:s'),
                'error_type' => $error_type,
                'message' => $exception,
                'file' => 'unknown',
                'line' => 'unknown',
                'trace' => 'N/A',
                'user_id' => $_SESSION['user_id'] ?? 'anonymous',
                'ip_address' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
                'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
                'request_uri' => $_SERVER['REQUEST_URI'] ?? 'unknown',
                'context' => $context
            ];
        } else {
            $log_entry = [
                'timestamp' => date('Y-m-d H:i:s'),
                'error_type' => $error_type,
                'message' => $exception->getMessage(),
                'file' => $exception->getFile(),
                'line' => $exception->getLine(),
                'trace' => $exception->getTraceAsString(),
                'user_id' => $_SESSION['user_id'] ?? 'anonymous',
                'ip_address' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
                'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
                'request_uri' => $_SERVER['REQUEST_URI'] ?? 'unknown',
                'context' => $context
            ];
        }
        
        $log_message = json_encode($log_entry, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT) . "\n";
        
        $log_dir = dirname(self::$error_log_file);
        if (!is_dir($log_dir)) {
            mkdir($log_dir, 0755, true);
        }
        
        file_put_contents(self::$error_log_file, $log_message, FILE_APPEND | LOCK_EX);
        
        if (is_string($exception)) {
            error_log("[$error_type] " . $exception);
        } else {
            error_log("[$error_type] " . $exception->getMessage() . " in " . $exception->getFile() . ":" . $exception->getLine());
        }
    }
    
    public static function handleDatabaseError($exception, $sql = '', $params = []) {
        $context = [
            'sql' => $sql,
            'params' => $params,
            'pdo_error_info' => $exception instanceof PDOException ? $exception->errorInfo : null
        ];
        
        return self::handleError($exception, 'database', $context);
    }
    
    public static function handleValidationError($message, $field = '', $value = '') {
        $exception = new Exception($message);
        $context = [
            'field' => $field,
            'value' => is_string($value) ? $value : json_encode($value)
        ];
        
        return self::handleError($exception, 'validation', $context);
    }
    
    public static function handleFileUploadError($exception, $file_info = []) {
        $context = [
            'file_info' => $file_info,
            'upload_errors' => [
                UPLOAD_ERR_INI_SIZE => 'Dosya boyutu çok büyük (php.ini)',
                UPLOAD_ERR_FORM_SIZE => 'Dosya boyutu çok büyük (form)',
                UPLOAD_ERR_PARTIAL => 'Dosya kısmen yüklendi',
                UPLOAD_ERR_NO_FILE => 'Dosya seçilmedi',
                UPLOAD_ERR_NO_TMP_DIR => 'Geçici dizin bulunamadı',
                UPLOAD_ERR_CANT_WRITE => 'Dosya yazılamadı',
                UPLOAD_ERR_EXTENSION => 'Dosya uzantısı engellendi'
            ]
        ];
        
        return self::handleError($exception, 'file_upload', $context);
    }
    
    public static function handleCSRFError() {
        $exception = new Exception('CSRF token validation failed');
        return self::handleError($exception, 'csrf');
    }
    
    public static function handleRateLimitError($attempts, $time_window) {
        $exception = new Exception("Rate limit exceeded: $attempts attempts in $time_window seconds");
        $context = [
            'attempts' => $attempts,
            'time_window' => $time_window
        ];
        
        return self::handleError($exception, 'rate_limit', $context);
    }
    
    public static function handleAuthorizationError($required_permission = '', $user_role = '') {
        $exception = new Exception('Access denied');
        $context = [
            'required_permission' => $required_permission,
            'user_role' => $user_role
        ];
        
        return self::handleError($exception, 'authorization', $context);
    }
    
    public static function cleanupLogs($days_to_keep = 30) {
        $log_file = self::$error_log_file;
        
        if (!file_exists($log_file)) {
            return;
        }
        
        $lines = file($log_file, FILE_IGNORE_NEW_LINES);
        $cutoff_date = date('Y-m-d', strtotime("-$days_to_keep days"));
        $filtered_lines = [];
        
        foreach ($lines as $line) {
            if (empty(trim($line))) continue;
            
            $log_data = json_decode($line, true);
            if ($log_data && isset($log_data['timestamp'])) {
                $log_date = substr($log_data['timestamp'], 0, 10);
                if ($log_date >= $cutoff_date) {
                    $filtered_lines[] = $line;
                }
            }
        }
        
        file_put_contents($log_file, implode("\n", $filtered_lines) . "\n");
    }
    
    public static function getErrorStats($days = 7) {
        $log_file = self::$error_log_file;
        
        if (!file_exists($log_file)) {
            return [
                'total_errors' => 0,
                'by_type' => [],
                'by_date' => [],
                'top_errors' => []
            ];
        }
        
        $content = file_get_contents($log_file);
        $cutoff_date = date('Y-m-d', strtotime("-$days days"));
        $stats = [
            'total_errors' => 0,
            'by_type' => [],
            'by_date' => [],
            'top_errors' => []
        ];
        
        $json_objects = [];
        $current_json = '';
        $lines = explode("\n", $content);
        
        foreach ($lines as $line) {
            $line = trim($line);
            if (empty($line)) continue;
            
            $current_json .= $line;
            
            if (substr($line, -1) === '}') {
                $log_data = json_decode($current_json, true);
                if ($log_data && isset($log_data['timestamp'])) {
                    $log_date = substr($log_data['timestamp'], 0, 10);
                    if ($log_date >= $cutoff_date) {
                        $stats['total_errors']++;
                        
                        $error_type = $log_data['error_type'] ?? 'unknown';
                        $stats['by_type'][$error_type] = ($stats['by_type'][$error_type] ?? 0) + 1;
                        
                        $stats['by_date'][$log_date] = ($stats['by_date'][$log_date] ?? 0) + 1;
                        
                        $error_message = $log_data['message'] ?? 'Unknown error';
                        $stats['top_errors'][$error_message] = ($stats['top_errors'][$error_message] ?? 0) + 1;
                    }
                }
                $current_json = '';
            }
        }
        
        arsort($stats['top_errors']);
        $stats['top_errors'] = array_slice($stats['top_errors'], 0, 10, true);
        
        return $stats;
    }
    
    public static function notifyCriticalError($exception, $error_type = 'critical') {
        
        $critical_log = __DIR__ . '/../logs/critical_errors.log';
        $log_entry = [
            'timestamp' => date('Y-m-d H:i:s'),
            'error_type' => $error_type,
            'message' => $exception->getMessage(),
            'file' => $exception->getFile(),
            'line' => $exception->getLine(),
            'severity' => 'CRITICAL'
        ];
        
        $log_message = json_encode($log_entry, JSON_UNESCAPED_UNICODE) . "\n";
        file_put_contents($critical_log, $log_message, FILE_APPEND | LOCK_EX);
    }
}

function globalExceptionHandler($exception) {
    ErrorHandler::logError($exception, 'uncaught');
    
    if (!defined('DEBUG_MODE') || !DEBUG_MODE) {
        http_response_code(500);
        include __DIR__ . '/../pages/error_500.php';
        exit();
    }
    
    throw $exception;
}

set_exception_handler('globalExceptionHandler');

function globalErrorHandler($severity, $message, $file, $line) {
    if (!(error_reporting() & $severity)) {
        return false;
    }
    
    $exception = new ErrorException($message, 0, $severity, $file, $line);
    ErrorHandler::logError($exception, 'php_error');
    
    return true;
}

set_error_handler('globalErrorHandler');
?>