<?php

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/balance_manager.php';

/**
 * DataCleanup Class
 * Hatalı verileri temizleyen ve düzelten sınıf
 */
class DataCleanup {
    private $database;
    private $balanceManager;
    private $logFile;
    
    public function __construct($database = null) {
        $this->database = $database ?? Database::getInstance();
        $this->balanceManager = new BalanceManager($this->database);
        $this->logFile = __DIR__ . '/../logs/cleanup.log';
        
        $logDir = dirname($this->logFile);
        if (!file_exists($logDir)) {
            mkdir($logDir, 0755, true);
        }
    }
    
    /**
     * Hatalı kayıtları tespit et
     * @return array Tespit edilen sorunlar
     */
    public function detectInconsistencies(): array {
        try {
            $this->log("Tutarsızlık tespiti başlatılıyor");
            
            $issues = [];
            
            // 1. Bakiye tutarsızlıkları
            $balanceIssues = $this->checkBalanceConsistency();
            if (!empty($balanceIssues)) {
                $issues['balance_inconsistencies'] = $balanceIssues;
                $this->log("Bakiye tutarsızlığı bulundu: " . count($balanceIssues) . " kayıt");
            }
            
            // 2. Negatif müşteri bakiyeleri
            $negativeCustomers = $this->findNegativeCustomerBalances();
            if (!empty($negativeCustomers)) {
                $issues['negative_customer_balances'] = $negativeCustomers;
                $this->log("Negatif müşteri bakiyesi bulundu: " . count($negativeCustomers) . " kayıt");
            }
            
            // 3. Orphaned kayıtlar
            $orphanedRecords = $this->findOrphanedRecords();
            if (!empty($orphanedRecords)) {
                $issues['orphaned_records'] = $orphanedRecords;
                $this->log("Orphaned kayıt bulundu: " . count($orphanedRecords) . " kayıt");
            }
            
            // 4. Overflow kayıtlar
            $overflowRecords = $this->findOverflowRecords();
            if (!empty($overflowRecords)) {
                $issues['overflow_records'] = $overflowRecords;
                $this->log("Overflow kayıt bulundu: " . count($overflowRecords) . " kayıt");
            }
            
            // 5. Satış tutarsızlıkları
            $salesIssues = $this->checkSalesConsistency();
            if (!empty($salesIssues)) {
                $issues['sales_inconsistencies'] = $salesIssues;
                $this->log("Satış tutarsızlığı bulundu: " . count($salesIssues) . " kayıt");
            }
            
            $this->log("Tutarsızlık tespiti tamamlandı: " . count($issues) . " kategori");
            
            return [
                'success' => true,
                'issues' => $issues,
                'total_issues' => array_sum(array_map('count', $issues)),
                'message' => 'Tutarsızlık tespiti tamamlandı'
            ];
            
        } catch (Exception $e) {
            $this->log("Tutarsızlık tespiti hatası: " . $e->getMessage(), 'ERROR');
            return [
                'success' => false,
                'message' => 'Tutarsızlık tespiti sırasında hata: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Overflow olan kayıtları bul
     * @return array Overflow kayıtlar
     */
    public function findOverflowRecords(): array {
        try {
            $overflowRecords = [];
            
            // DECIMAL(10,2) maksimum değeri: 99999999.99
            $maxValue = 99999999.99;
            
            // cariler tablosu
            $sql = "SELECT cari_id, firma_adi, cari_bakiye, bakiye 
                    FROM cariler 
                    WHERE ABS(cari_bakiye) >= ? OR ABS(bakiye) >= ?";
            $records = $this->database->fetchAll($sql, [$maxValue, $maxValue]);
            if (!empty($records)) {
                $overflowRecords['cariler'] = $records;
            }
            
            // cari_hareketler tablosu
            $sql = "SELECT hareket_id, cari_id, tutar 
                    FROM cari_hareketler 
                    WHERE ABS(tutar) >= ?";
            $records = $this->database->fetchAll($sql, [$maxValue]);
            if (!empty($records)) {
                $overflowRecords['cari_hareketler'] = $records;
            }
            
            // satislar tablosu
            $sql = "SELECT satis_id, toplam_tutar, odenen_tutar, kalan_tutar 
                    FROM satislar 
                    WHERE ABS(toplam_tutar) >= ? OR ABS(odenen_tutar) >= ? OR ABS(kalan_tutar) >= ?";
            $records = $this->database->fetchAll($sql, [$maxValue, $maxValue, $maxValue]);
            if (!empty($records)) {
                $overflowRecords['satislar'] = $records;
            }
            
            return $overflowRecords;
            
        } catch (Exception $e) {
            $this->log("Overflow kayıt bulma hatası: " . $e->getMessage(), 'ERROR');
            return [];
        }
    }
    
    /**
     * Foreign key bağımlılıklarını çöz
     * @return array İşlem sonucu
     */
    public function resolveDependencies(): array {
        try {
            $this->log("Foreign key bağımlılıkları çözülüyor");
            
            // Foreign key constraint'leri geçici olarak devre dışı bırak
            $this->database->execute("SET FOREIGN_KEY_CHECKS = 0");
            $this->log("Foreign key checks devre dışı bırakıldı");
            
            return [
                'success' => true,
                'message' => 'Foreign key constraint\'leri devre dışı bırakıldı'
            ];
            
        } catch (Exception $e) {
            $this->log("Bağımlılık çözme hatası: " . $e->getMessage(), 'ERROR');
            return [
                'success' => false,
                'message' => 'Bağımlılık çözme sırasında hata: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Foreign key constraint'leri yeniden aktif et
     * @return array İşlem sonucu
     */
    public function enableConstraints(): array {
        try {
            $this->log("Foreign key constraint'leri aktif ediliyor");
            
            $this->database->execute("SET FOREIGN_KEY_CHECKS = 1");
            $this->log("Foreign key checks aktif edildi");
            
            return [
                'success' => true,
                'message' => 'Foreign key constraint\'leri aktif edildi'
            ];
            
        } catch (Exception $e) {
            $this->log("Constraint aktif etme hatası: " . $e->getMessage(), 'ERROR');
            return [
                'success' => false,
                'message' => 'Constraint aktif etme sırasında hata: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Hatalı kayıtları temizle
     * @param array $recordIds Temizlenecek kayıt ID'leri
     * @return array İşlem sonucu
     */
    public function cleanupRecords(array $recordIds): array {
        try {
            $this->log("Kayıt temizleme başlatılıyor");
            
            $this->database->beginTransaction();
            
            $cleanedCount = 0;
            
            // Örnek: cari_hareketler tablosundan temizlik
            if (isset($recordIds['cari_hareketler'])) {
                foreach ($recordIds['cari_hareketler'] as $id) {
                    // Önce yedeğe al
                    $sql = "SELECT * FROM cari_hareketler WHERE hareket_id = ?";
                    $record = $this->database->fetch($sql, [$id]);
                    
                    if ($record) {
                        // Yedekleme dosyasına yaz
                        $this->backupRecord('cari_hareketler', $record);
                        
                        // Kaydı sil
                        $sql = "DELETE FROM cari_hareketler WHERE hareket_id = ?";
                        $this->database->execute($sql, [$id]);
                        $cleanedCount++;
                    }
                }
            }
            
            $this->database->commit();
            
            $this->log("Kayıt temizleme tamamlandı: $cleanedCount kayıt temizlendi");
            
            return [
                'success' => true,
                'cleaned_count' => $cleanedCount,
                'message' => 'Kayıtlar başarıyla temizlendi'
            ];
            
        } catch (Exception $e) {
            $this->database->rollback();
            $this->log("Kayıt temizleme hatası: " . $e->getMessage(), 'ERROR');
            return [
                'success' => false,
                'message' => 'Kayıt temizleme sırasında hata: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Temizlik raporu oluştur
     * @return array Rapor
     */
    public function generateCleanupReport(): array {
        try {
            $this->log("Temizlik raporu oluşturuluyor");
            
            $report = [];
            
            // Mevcut durum analizi
            $report['current_state'] = [
                'total_cariler' => $this->database->fetch("SELECT COUNT(*) as count FROM cariler")['count'],
                'total_hareketler' => $this->database->fetch("SELECT COUNT(*) as count FROM cari_hareketler")['count'],
                'total_satislar' => $this->database->fetch("SELECT COUNT(*) as count FROM satislar")['count'],
                'total_stok_hareketleri' => $this->database->fetch("SELECT COUNT(*) as count FROM stok_hareketleri")['count']
            ];
            
            // Tutarsızlıklar
            $report['inconsistencies'] = $this->detectInconsistencies();
            
            // Bakiye özeti
            $report['balance_summary'] = $this->balanceManager->getBalanceSummary();
            
            $this->log("Temizlik raporu oluşturuldu");
            
            return [
                'success' => true,
                'report' => $report,
                'message' => 'Temizlik raporu başarıyla oluşturuldu'
            ];
            
        } catch (Exception $e) {
            $this->log("Rapor oluşturma hatası: " . $e->getMessage(), 'ERROR');
            return [
                'success' => false,
                'message' => 'Rapor oluşturma sırasında hata: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Bakiye tutarlılığını kontrol et
     * @return array Tutarsız kayıtlar
     */
    private function checkBalanceConsistency(): array {
        return $this->balanceManager->checkBalanceConsistency();
    }
    
    /**
     * Negatif müşteri bakiyelerini bul
     * @return array Negatif bakiyeli müşteriler
     */
    private function findNegativeCustomerBalances(): array {
        try {
            $sql = "SELECT cari_id, firma_adi, cari_adi, cari_bakiye, bakiye 
                    FROM cariler 
                    WHERE cari_tipi = 'musteri' AND (cari_bakiye < 0 OR bakiye > 0)";
            return $this->database->fetchAll($sql);
        } catch (Exception $e) {
            $this->log("Negatif bakiye bulma hatası: " . $e->getMessage(), 'ERROR');
            return [];
        }
    }
    
    /**
     * Orphaned kayıtları bul
     * @return array Orphaned kayıtlar
     */
    private function findOrphanedRecords(): array {
        try {
            $orphaned = [];
            
            // cari_hareketler'de olmayan cari_id'ler
            $sql = "SELECT ch.hareket_id, ch.cari_id 
                    FROM cari_hareketler ch 
                    LEFT JOIN cariler c ON ch.cari_id = c.cari_id 
                    WHERE c.cari_id IS NULL";
            $records = $this->database->fetchAll($sql);
            if (!empty($records)) {
                $orphaned['cari_hareketler'] = $records;
            }
            
            // satislar'da olmayan cari_id'ler
            $sql = "SELECT s.satis_id, s.cari_id 
                    FROM satislar s 
                    LEFT JOIN cariler c ON s.cari_id = c.cari_id 
                    WHERE c.cari_id IS NULL";
            $records = $this->database->fetchAll($sql);
            if (!empty($records)) {
                $orphaned['satislar'] = $records;
            }
            
            // satis_detay'da olmayan satis_id'ler
            $sql = "SELECT sd.satis_detay_id, sd.satis_id 
                    FROM satis_detay sd 
                    LEFT JOIN satislar s ON sd.satis_id = s.satis_id 
                    WHERE s.satis_id IS NULL";
            $records = $this->database->fetchAll($sql);
            if (!empty($records)) {
                $orphaned['satis_detay'] = $records;
            }
            
            return $orphaned;
            
        } catch (Exception $e) {
            $this->log("Orphaned kayıt bulma hatası: " . $e->getMessage(), 'ERROR');
            return [];
        }
    }
    
    /**
     * Satış tutarsızlıklarını kontrol et
     * @return array Tutarsız satışlar
     */
    private function checkSalesConsistency(): array {
        try {
            $issues = [];
            
            // Kalan tutar hesaplaması yanlış olan satışlar
            $sql = "SELECT satis_id, toplam_tutar, odenen_tutar, kalan_tutar,
                           (toplam_tutar - odenen_tutar) as calculated_kalan
                    FROM satislar 
                    WHERE ABS(kalan_tutar - (toplam_tutar - odenen_tutar)) > 0.01";
            $records = $this->database->fetchAll($sql);
            if (!empty($records)) {
                $issues['incorrect_kalan_tutar'] = $records;
            }
            
            // Negatif tutarlı satışlar
            $sql = "SELECT satis_id, toplam_tutar, odenen_tutar, kalan_tutar 
                    FROM satislar 
                    WHERE toplam_tutar < 0 OR odenen_tutar < 0 OR kalan_tutar < 0";
            $records = $this->database->fetchAll($sql);
            if (!empty($records)) {
                $issues['negative_amounts'] = $records;
            }
            
            return $issues;
            
        } catch (Exception $e) {
            $this->log("Satış tutarsızlık kontrolü hatası: " . $e->getMessage(), 'ERROR');
            return [];
        }
    }
    
    /**
     * Kaydı yedekle
     * @param string $table Tablo adı
     * @param array $record Kayıt
     */
    private function backupRecord(string $table, array $record): void {
        $backupDir = __DIR__ . '/../backups/deleted_records/';
        if (!file_exists($backupDir)) {
            mkdir($backupDir, 0755, true);
        }
        
        $backupFile = $backupDir . $table . '_' . date('Y-m-d') . '.json';
        
        $existingData = [];
        if (file_exists($backupFile)) {
            $existingData = json_decode(file_get_contents($backupFile), true) ?? [];
        }
        
        $existingData[] = [
            'timestamp' => date('Y-m-d H:i:s'),
            'record' => $record
        ];
        
        file_put_contents($backupFile, json_encode($existingData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    }
    
    /**
     * Log mesajı yaz
     * @param string $message Mesaj
     * @param string $level Log seviyesi
     */
    private function log(string $message, string $level = 'INFO'): void {
        $timestamp = date('Y-m-d H:i:s');
        $logMessage = "[$timestamp] [$level] $message\n";
        file_put_contents($this->logFile, $logMessage, FILE_APPEND);
    }
}

?>
