<?php

require_once __DIR__ . '/config.php';

class Database {
    private static $instance = null;
    private $host = 'localhost';
    private $db_name = 'depo_stok_takip';
    private $username = 'root';
    private $password = '';
    private $charset = 'utf8mb4';
    private $pdo;
    private $error;

    public function __construct() {
        $this->connect();
    }

    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function connect() {
        $dsn = "mysql:host={$this->host};dbname={$this->db_name};charset={$this->charset}";
        
        $options = [
            PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES   => false,
            PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci"
        ];

        try {
            $this->pdo = new PDO($dsn, $this->username, $this->password, $options);
        } catch (PDOException $e) {
            $this->error = $e->getMessage();
            error_log("Database connection error: " . $this->error);
            throw new Exception("Veritabanı bağlantı hatası");
        }
    }

    public function getConnection() {
        return $this->pdo;
    }

    public function getError() {
        return $this->error;
    }

    public function execute($sql, $params = []) {
        try {
            $stmt = $this->pdo->prepare($sql);
            return $stmt->execute($params);
        } catch (PDOException $e) {
            error_log("Database execute error: " . $e->getMessage() . " SQL: " . $sql);
            throw new Exception("Veritabanı işlem hatası");
        }
    }

    public function fetch($sql, $params = []) {
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            return $stmt->fetch();
        } catch (PDOException $e) {
            error_log("Database fetch error: " . $e->getMessage() . " SQL: " . $sql);
            throw new Exception("Veri çekme hatası");
        }
    }

    public function fetchAll($sql, $params = []) {
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            return $stmt->fetchAll();
        } catch (PDOException $e) {
            error_log("Database fetchAll error: " . $e->getMessage() . " SQL: " . $sql);
            throw new Exception("Veri çekme hatası");
        }
    }

    public function lastInsertId() {
        return $this->pdo->lastInsertId();
    }

    public function rowCount($sql, $params = []) {
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            return $stmt->rowCount();
        } catch (PDOException $e) {
            error_log("Database rowCount error: " . $e->getMessage() . " SQL: " . $sql);
            throw new Exception("Satır sayısı alma hatası");
        }
    }

    public function beginTransaction() {
        try {
            if (!$this->pdo->inTransaction()) {
                return $this->pdo->beginTransaction();
            }
            return true; // Zaten transaction içindeyiz
        } catch (PDOException $e) {
            error_log("Transaction başlatma hatası: " . $e->getMessage());
            throw new Exception("Transaction başlatılamadı");
        }
    }

    public function commit() {
        try {
            if ($this->pdo->inTransaction()) {
                return $this->pdo->commit();
            }
            return true; // Zaten commit edilmiş
        } catch (PDOException $e) {
            error_log("Transaction commit hatası: " . $e->getMessage());
            throw new Exception("Transaction commit edilemedi");
        }
    }

    public function rollback() {
        try {
            if ($this->pdo->inTransaction()) {
                return $this->pdo->rollback();
            }
            return true; // Zaten rollback edilmiş
        } catch (PDOException $e) {
            error_log("Transaction rollback hatası: " . $e->getMessage());
            throw new Exception("Transaction rollback edilemedi");
        }
    }

    public function inTransaction() {
        return $this->pdo->inTransaction();
    }

    public function safeTransaction($callback) {
        $wasInTransaction = $this->inTransaction();
        
        try {
            if (!$wasInTransaction) {
                $this->beginTransaction();
            }
            
            $result = $callback($this);
            
            if (!$wasInTransaction) {
                $this->commit();
            }
            
            return $result;
            
        } catch (Exception $e) {
            if (!$wasInTransaction && $this->inTransaction()) {
                $this->rollback();
            }
            throw $e;
        }
    }

    public function quote($string) {
        return $this->pdo->quote($string);
    }

    public function prepare($sql) {
        return $this->pdo->prepare($sql);
    }

    public function query($sql) {
        return $this->pdo->query($sql);
    }

    public function exec($sql) {
        return $this->pdo->exec($sql);
    }

    public function getAttribute($attribute) {
        return $this->pdo->getAttribute($attribute);
    }

    public function setAttribute($attribute, $value) {
        return $this->pdo->setAttribute($attribute, $value);
    }

    public function getAvailableDrivers() {
        return $this->pdo->getAvailableDrivers();
    }

    public function errorCode() {
        return $this->pdo->errorCode();
    }

    public function errorInfo() {
        return $this->pdo->errorInfo();
    }

    public function __destruct() {
        $this->pdo = null;
    }
}

$database = Database::getInstance();

?>